"""Creates an Obstructions coverage hidden component."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import write_display_option_ids
from xms.data_objects.parameters import Component

# 4. Local modules
from xms.srh.components import obstruction_component as obc
from xms.srh.data.obstruction_data import ObstructionData


def build_obstructions_component(obs_data, cov_uuid, comp_dir, pier_ids, deck_ids, new_comp_uuid=None):
    """Create the obstructions coverage's hidden component and data.

    Args:
        obs_data (:obj:`xarray.Dataset`): The obstructions component dataset
        cov_uuid (:obj:`str`): UUID of the obstructions coverage geometry
        comp_dir (:obj:`str`): Path to the XMS temp components folder (where the component will be created).
        pier_ids (:obj:`dict`): Pier feature point id to component id
        deck_ids (:obj:`dict`): Deck feature arc id to component id
        new_comp_uuid (:obj:`str`): UUID of the new BC component. Useful for testing, usually just want
            randomly generated.

    Returns:
        (:obj:`xms.data_objects.parameters.Component`): data_object for the new Obstructions component
    """
    # Create a new UUID and folder for the component data
    if new_comp_uuid:  # Testing, use hard-coded UUID
        comp_uuid = new_comp_uuid
    else:  # pragma: no cover
        comp_uuid = str(uuid.uuid4())
    obs_comp_dir = os.path.join(comp_dir, comp_uuid)
    os.makedirs(obs_comp_dir, exist_ok=True)
    obs_main_file = os.path.join(obs_comp_dir, 'obstruction_comp.nc')

    # Create the data_object Component to send back to SMS
    obs_do_comp = Component(
        comp_uuid=comp_uuid, main_file=obs_main_file, model_name='SRH-2D', unique_name='Obstruction_Component'
    )

    data = ObstructionData(obs_main_file)
    data.obstruction_data = obs_data
    data.info.attrs['cov_uuid'] = cov_uuid
    data.commit()

    # Write component id and att ids to a file so we can initialize them in get_initial_display_options
    id_file = os.path.join(obs_comp_dir, obc.INITIAL_OBS_PT_ATT_ID_FILE)
    write_display_option_ids(id_file, list(pier_ids.keys()))
    id_file = os.path.join(obs_comp_dir, obc.INITIAL_OBS_PT_COMP_ID_FILE)
    write_display_option_ids(id_file, list(pier_ids.values()))
    id_file = os.path.join(obs_comp_dir, obc.INITIAL_OBS_ARC_ATT_ID_FILE)
    write_display_option_ids(id_file, list(deck_ids.keys()))
    id_file = os.path.join(obs_comp_dir, obc.INITIAL_OBS_ARC_COMP_ID_FILE)
    write_display_option_ids(id_file, list(deck_ids.values()))

    return obs_do_comp
