"""Module for exporting input files for an SRH-2D parameter run."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.core.filesystem import filesystem
from xms.guipy.dialogs import treeitem_selector_datasets

# 4. Local modules
from xms.srh.components.parameters.parameter_data import ParameterData
from xms.srh.components.parameters.parameters_manager import ParametersManager
from xms.srh.file_io import io_util

QUOTE_STRIP = '"��\''


class ParametersExporter:
    """Handles exporting simulations that use parameters."""
    def __init__(self, vwse_datasets):
        """Initializes the class.

        Args:
            vwse_datasets (:obj:`list[list[float]]`): List of VWSE datasets. Will be popped of in order of
                the runs that use datasets for initial condition (testing)
        """
        self.writer = None  # FileWriter class
        self.param_data = None  # ParameterData
        self.vwse_dsets = vwse_datasets
        self.logger = logging.getLogger('xms.srh')

    def export(self, writer):
        """Export the part of the simulation that can be parameterized.

        Args:
            writer (:obj:`FileWriter`): Worker that contains data needed for export
        """
        self.writer = writer
        param_dict = ParametersManager.read_parameter_file(writer.sim_component.main_file)
        self.param_data = ParameterData.from_dict(param_dict)

        # Export non-parameter stuff
        # self.writer.export_geom()
        # self.writer.export_materials()
        # self.writer.export_sed_materials()
        # self.writer.export_monitor_points()

        self._export_runs()

    def _export_runs(self):
        """Exports the runs to subfolders."""
        out_dir = self.writer.out_dir

        # Convert xy_series list to a dict
        xy_series = {series[0]: (series[1], series[2]) for series in self.param_data.xy_series}

        run_count = self.param_data.run_count
        for run in range(run_count):
            run_name = self.param_data.runs['Run'][run]

            self.logger.info(f'Writing run {run + 1}: "{run_name}".')

            # Change the case name to be the run name
            self.writer.sim_component.data.hydro.case_name = run_name

            # Make the subfolder and change the self.writer.out_dir
            self.writer.out_dir = os.path.join(out_dir, run_name)
            filesystem.make_or_clear_dir(self.writer.out_dir)

            # Swap in the parameter values
            self.swap_in_param_values(run, xy_series)

            # Save .srhhydro file and post file to the subfolder
            self.writer.do_export()
            # self.writer.export_wse_dataset()
            # self.writer.export_hydro(run_name=run_name)
            # self.writer.export_srh_post(using_parameters=True)

            self.logger.info(f'Done writing run {run + 1}.')

    def swap_in_param_values(self, run, xy_series):
        """Replaces existing values with parameter values for the given run.

        Args:
            run (:obj:`int`): The run index
            xy_series (:obj:`dict`): Dict of the xy series
        """
        for parameter in self.param_data.params:
            if not parameter.use:
                continue  # Skip this parameter - not in use

            # Get the value for this parameter for this run.
            values = parameter.get_values(self.param_data.runs, run)
            if parameter.type == "Manning's N":
                self.set_material_mannings_n(parameter.id_string(), values[0])
            elif parameter.type == 'Time step':
                self.writer.sim_component.data.hydro.time_step = values[0]
            elif parameter.type == 'Initial condition':
                self.set_initial_condition(values[0], values[1], run)
            else:  # bcs
                self.set_bcs(parameter.id_string(), values, xy_series)

    def set_initial_condition(self, _type, value, run):
        """Sets the initial condition based on the parameter.

        Args:
            _type (:obj:`str`): Initial condition type:
                Dry, Initial Water Surface Elevation, Automatic, Restart File, Water Surface Elevation Dataset
            value (:obj:`str`): parameter value (constant for Initial Water Surface Elevation, file path for
                Restart File, uuid for Water Surface Elevation Dataset)
            run (:obj:`int`): Run number.
        """
        if not _type or value is None:  # pragma: no cover
            self.logger.error(f'Invalid parameter value for initial condition for run {run + 1}.')
            return

        if type(value) is str:
            value.strip(QUOTE_STRIP)  # Strip all kinds of quotes from front and back

        if _type == 'Dry':
            self.writer.sim_component.data.hydro.initial_condition = 'Dry'
        elif _type == 'Initial Water Surface Elevation' and io_util.is_float(value):
            self.writer.sim_component.data.hydro.initial_condition = 'Initial Water Surface Elevation'
            self.writer.sim_component.data.hydro.initial_water_surface_elevation = float(value)
        elif _type == 'Automatic':
            self.writer.sim_component.data.hydro.initial_condition = 'Automatic'
        elif _type == 'Restart File':
            self.writer.sim_component.data.hydro.initial_condition = 'Restart File'
            self.writer.sim_component.data.hydro.restart_file = value
        elif _type == 'Water Surface Elevation Dataset' and value:
            dset_uuid, _ = treeitem_selector_datasets.uuid_and_time_step_index_from_string(value)
            if io_util.is_valid_uuid(dset_uuid):
                self.writer.sim_component.data.hydro.initial_condition = 'Water Surface Elevation Dataset'

                # Follow path to get item and its uuid
                if self.vwse_dsets:  # Testing
                    self.writer.wse_data = self.vwse_dsets.pop()
                else:  # pragma: no cover
                    self.writer.sim_component.data.hydro.water_surface_elevation_dataset = value
                    self.writer.sim_query_helper.get_wse_dataset()
                    self.writer.wse_data = self.writer.sim_query_helper.wse_dataset

    def set_material_mannings_n(self, _id, value):
        """Sets the Manning's N value of the material to the parameter value.

        Args:
            _id (:obj:`str`): id string from params table.
            value (:obj:`float or str`): The parameter value.
        """
        mat_idx = int(_id[4:])
        self.writer.coverage_mapper.material_mannings[mat_idx] = value

    def set_bcs(self, _id, values, xy_series):
        """Swap parameters into the bc data.

        Args:
            _id (:obj:`str`): id string from params table.
            values (:obj:`list`): List of the parameter values: 0='Constant' or 'Transient', 1=float, 2=xy series name
            xy_series (:obj:`dict`): Dict of defined xy series.
        """
        is_bc, bc_length = ParameterData.is_bc(_id)
        if is_bc:
            arc_id = int(_id[bc_length:])
            bc_param = self.writer.coverage_mapper.bc_arc_id_to_bc_param[arc_id]
            if bc_param.bc_type.startswith('Inlet-Q'):
                self._set_bc_data(
                    values[0], values[1], values[2], xy_series, bc_param.inlet_q,
                    ['discharge_option', 'constant_q', 'time_series_q']
                )
            elif bc_param.bc_type.startswith('Exit-H'):
                self._set_bc_data(
                    values[0], values[1], values[2], xy_series, bc_param.exit_h,
                    ['water_surface_elevation_option', 'constant_wse', 'time_series_wse', 'rating_curve']
                )
            elif bc_param.bc_type.startswith('Internal sink'):
                self._set_bc_data(
                    values[0], values[1], values[2], xy_series, bc_param.internal_sink,
                    ['sink_flow_type', 'constant_q', 'time_series_q', 'rating_curve']
                )

    @staticmethod
    def _set_bc_data(transient_word, steady_state_value, xy_series_name, xy_series, bc_param, attributes):
        """Sets the bc data to the parameter value, either constant or xy series.

        Args:
            transient_word (:obj:`str`): "Transient" or "Constant".
            steady_state_value (:obj:`float`): The value if it is constant.
            xy_series_name (:obj:`str`): The name of the xy series if it is transient.
            xy_series (:obj:`dict`): Dict of defined xy series.
            bc_param: BcDataInletQ, BcDataExitQ, or BcDataInternalSink
            attributes (:obj:`list[str]`): List of the bc attributes where the data is stored.
                attributes[0] = constant or time series option, e.g. 'discharge_option' or 'sink_flow_type'
                attributes[1] = constant value, e.g. 'constant_q' or 'constant_wse'
                attributes[2] = time series, e.g. 'time_series_q' or 'time_series_wse'
                attributes[3] = rating curve, e.g. 'rating_curve'
        """
        constant_or_ts_att = attributes[0]
        constant_att = attributes[1]
        ts_att = attributes[2]
        rating_curve_att = attributes[3] if len(attributes) > 3 else None

        if transient_word == 'Transient' and xy_series_name in xy_series:  # value is the name of an xy series
            if xy_series[xy_series_name][1] and rating_curve_att:
                # xy series is a rating curve and the bc has a rating curve att
                setattr(bc_param, constant_or_ts_att, 'Rating curve')
                column_names = list(getattr(bc_param, rating_curve_att).columns.values)
                df = pd.DataFrame(
                    {
                        column_names[0]: xy_series[xy_series_name][0]['X'],
                        column_names[1]: xy_series[xy_series_name][0]['Y']
                    }
                )
                setattr(bc_param, rating_curve_att, df)
            else:
                # xy series is not a rating curve, or the bc doesn't have a rating curve att
                setattr(bc_param, constant_or_ts_att, 'Time series')
                column_names = list(getattr(bc_param, ts_att).columns.values)
                df = pd.DataFrame(
                    {
                        column_names[0]: xy_series[xy_series_name][0]['X'],
                        column_names[1]: xy_series[xy_series_name][0]['Y']
                    }
                )
                setattr(bc_param, ts_att, df)
        elif transient_word == 'Constant':  # value is not an xy series but is a number
            setattr(bc_param, constant_or_ts_att, 'Constant')
            setattr(bc_param, constant_att, float(steady_state_value))
