"""The Generate Floodway Evaluation Lines dialog."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtWidgets import QDialog, QDialogButtonBox

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.treeitem_selector_datasets import TreeItemSelectorDatasetsDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.srh.floodway.generate_floodway_evaluation_lines import GenerateFloodwayEvaluationLines
from xms.srh.gui.generate_floodway_evaluation_lines_dialog_ui import Ui_GenerateFloodwayEvaluationLinesDialog
from xms.srh.gui.manning_n_dialog import str_to_float


class GenerateFloodwayEvaluationLinesDialog(XmsDlg):
    """The Generate Floodway Evaluation Lines dialog."""
    def __init__(self, parent=None, help_url=None, query=None):
        """Initializes the dialog.

        Args:
            parent (:obj:`QObject`): The Qt parent object
            help_url (:obj:`string`): The url that we want the help button to bring up
            query (:obj:`Query`): XMS interprocess communication object
        """
        super().__init__(parent, 'xms.srh.gui.generate_floodway_evaluation_lines_dialog')

        self.ui = Ui_GenerateFloodwayEvaluationLinesDialog()
        self.ui.setupUi(self)
        self.query = query
        self.generator = GenerateFloodwayEvaluationLines(query)

        # example of help URL
        # help_url = "https://www.xmswiki.com/wiki/SMS:Display_Options"
        if help_url:
            self.help_url = help_url
        else:
            self.ui.btn_help.hide()

        self.setup_controls()
        self.setup_connections()
        self.toggle_intersect_contours()

        self.check_if_ok()

    def setup_controls(self):
        """Set up the controls for the dialog."""
        # Minimum value
        dbl_valid = QxDoubleValidator(parent=self)
        self.ui.edt_min_value.setValidator(dbl_valid)
        initial_string = str(0.0)
        self.ui.edt_min_value.setText(initial_string)
        # Maximum value
        self.ui.edt_max_value.setValidator(dbl_valid)
        self.ui.edt_max_value.setText(initial_string)
        # Interval
        self.ui.edt_interval.setValidator(dbl_valid)
        self.ui.edt_interval.setText(initial_string)
        # Threshold
        self.ui.edt_threshold.setValidator(dbl_valid)
        self.ui.edt_threshold.setText(initial_string)

        # disable the dataset data until the user selects a dataset
        self.ui.txt_min_value.setEnabled(False)
        self.ui.txt_max_value.setEnabled(False)
        self.ui.tog_specify_range.setEnabled(False)
        self.ui.txt_min_value_txt.setEnabled(False)
        self.ui.edt_min_value.setEnabled(False)
        self.ui.txt_max_value_txt.setEnabled(False)
        self.ui.edt_max_value.setEnabled(False)
        self.ui.txt_interval.setEnabled(False)
        self.ui.edt_interval.setEnabled(False)

        # set inital values
        self.ui.edt_output.setText('Floodway Evaluation Lines')
        self.ui.edt_interval.setText('1.0')
        self.ui.edt_threshold.setText('100.0')

    def setup_connections(self):
        """Setup signals and slots for the dialog's controls."""
        self.ui.tog_intersect_contours.clicked.connect(self.toggle_intersect_contours)
        self.ui.btn_help.clicked.connect(self.help_requested)

        self.ui.btn_wse.clicked.connect(self.select_dataset)
        self.ui.btn_centerline.clicked.connect(self.select_coverage)

        self.ui.tog_specify_range.clicked.connect(self.toggle_specify_range)
        self.ui.edt_interval.textChanged.connect(self.get_contour_values)
        self.ui.edt_min_value.textChanged.connect(self.get_contour_values)
        self.ui.edt_max_value.textChanged.connect(self.get_contour_values)

    def select_coverage(self):
        """Dialog to select a coverage."""
        dialog = TreeItemSelectorDlg(
            title='Select Centerline Coverage',
            pe_tree=self.generator.pe_tree,
            target_type=xmd.CoverageItem,
            previous_selection=self.generator.cl_uuid,
            show_root=True,
            parent=self
        )

        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.generator.select_cl_coverage(dialog.get_selected_item_uuid())
            self.ui.txt_centerline_selected.setText(self.generator.txt_centerline_selected)
        self.check_if_ok()

    def select_dataset(self):
        """Launches the select dataset dialog and sets the cross section values."""
        previous_selection = self.generator.wse_reader.uuid if self.generator.wse_reader else ''
        dialog = TreeItemSelectorDatasetsDlg(
            title='Select WSE Dataset',
            pe_tree=self.generator.grid_tree,
            selected_dataset=previous_selection,
            selected_time_step=self.generator.wse_ts,
            query=self.query,
            show_root=True,
            parent=self
        )

        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.generator.wse_ts = dialog.get_selected_time_step_index()
            dataset_uuid = dialog.get_selected_item_uuid()
            self.generator.select_dataset(dataset_uuid)
            valid = True if dataset_uuid else False
            if valid:
                # Get Min and Max values from the dataset
                self.ui.txt_min_value.setText(f'Minimum value: {self.generator.min_val}')
                self.ui.txt_max_value.setText(f'Maximum value: {self.generator.max_val}')

                self.ui.edt_min_value.setText(f'{self.generator.min_val}')
                self.ui.edt_max_value.setText(f'{self.generator.max_val}')

            self.ui.txt_wse_selected.setText(self.generator.txt_wse_selected)

            # enable dataset controls
            self.ui.txt_min_value.setEnabled(valid)
            self.ui.txt_max_value.setEnabled(valid)
            self.ui.tog_specify_range.setEnabled(valid)
            self.ui.tog_specify_range.setChecked(valid)
            self.ui.txt_interval.setEnabled(valid)
            self.ui.edt_interval.setEnabled(valid)

        self.toggle_specify_range()
        self.check_if_ok()

    def toggle_specify_range(self):
        """Update widget dependencies when the 'Specify range' checkbox is toggled."""
        enable = self.ui.tog_specify_range.isEnabled() and self.ui.tog_specify_range.isChecked()
        self.ui.txt_min_value_txt.setEnabled(enable)
        self.ui.edt_min_value.setEnabled(enable)
        self.ui.txt_max_value_txt.setEnabled(enable)
        self.ui.edt_max_value.setEnabled(enable)
        self.check_if_ok()

    def toggle_intersect_contours(self):
        """Update widget dependencies when the 'Connect all contours' checkbox is toggled."""
        enable = self.ui.tog_intersect_contours.isChecked()
        self.ui.txt_centerline_coverage.setEnabled(enable)
        self.ui.btn_centerline.setEnabled(enable)
        self.ui.txt_centerline_selected.setEnabled(enable)
        self.check_if_ok()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Create the Floodway Evaluation Lines coverage and send back to XMS when dialog is accepted."""
        # Get the grid from the simulation
        self.generator.threshold = str_to_float(self.ui.edt_threshold.text(), 0.0)
        self.generator.output_name = self.ui.edt_output.text()
        self.generator.intersect_contours = self.ui.tog_intersect_contours.isChecked()
        self.get_contour_values()

        super().accept()

    def get_contour_values(self):
        """Compute a list of contour values from the dialog input.

        Returns:
            (:obj:`list[float]`): See description
        """
        if self.ui.tog_specify_range.isChecked():
            self.generator.min_val = str_to_float(self.ui.edt_min_value.text(), self.generator.min_val)
            self.generator.max_val = str_to_float(self.ui.edt_max_value.text(), self.generator.max_val)
        self.generator.interval = str_to_float(self.ui.edt_interval.text(), 1.0)

        contour_values = self.generator.get_contour_values()

        self.ui.txt_max_num_contours.setText(f'Maximum number of contours: {self.generator.max_num_contours}')
        self.check_if_ok()
        return contour_values

    def check_if_ok(self):
        """Check if the 'OK' button should be enabled."""
        enable_ok = True
        need_centerline = self.ui.tog_intersect_contours.isChecked()
        if need_centerline and not self.generator.selected_centerline_cov:
            enable_ok = False
        if not self.generator.selected_dataset:
            enable_ok = False
        if self.generator.max_num_contours < 1:
            enable_ok = False

        if enable_ok:
            self.ui.buttonBox.button(QDialogButtonBox.Ok).setEnabled(True)
        else:
            self.ui.buttonBox.button(QDialogButtonBox.Ok).setEnabled(False)
