"""Dialog for specifying Manning's N calculation inputs."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import csv
import io
import os
import webbrowser

# 2. Third party modules
from matplotlib.backends.backend_qt5agg import FigureCanvas
from matplotlib.figure import Figure
from PySide2 import QtCore, QtGui
from PySide2.QtGui import Qt
from PySide2.QtWidgets import QApplication, QDialogButtonBox, QSplitter, QTableWidgetItem

# 3. Aquaveo modules
from xms.guipy.dialogs.message_box import message_with_n_buttons, message_with_ok
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.srh.gui.manning_n_dialog_ui import Ui_ManningNDialog
from xms.srh.manning_n.manning_n_data import ManningNData


def str_to_float(s, default):
    """Convert a string value to a float.

    Args:
        s (:obj:`str`): The string to convert
        default (:obj:`float`): Default value to return if conversion fails

    Returns:
        (:obj:`float`): The input string cast to a float or the specified default value if conversion fails
    """
    try:
        result = float(s)
        return result
    except ValueError:
        pass
    return default


class ManningNDialog(XmsDlg):
    """Dialog for specifying Manning's N calculation inputs."""
    def __init__(self, xms_getter, option=None, units=None, in_values=None, parent=None, previous_input=None):
        """Initializes the dialog.

        Args:
            xms_getter (:obj:`xms.srh.gui.manning_xms_getter.ManningXmsGetter`): Handles getting data from xms
            option (:obj:`str`): The BC type (Constant, Time series, or Rating curve)
            units (:obj:`str`): The type of units for units labels (U.S. Customary Units or SI Units (Metric))
            in_values (:obj:`list`): The input values (hours for time series, flows for rating curve)
            parent (:obj:`QObject`): The parent.
            previous_input (:obj:`dict`): Previous user input for the dialog
        """
        super().__init__(parent, 'xms.srh.gui.manning_n_dialog')

        # Try to get the previously selected dataset first. If it no longer exists, do not restore
        # previously entered inputs.
        self.xms_getter = xms_getter
        self.xms_getter.parent_dlg = self
        found_previous_dataset = False
        if previous_input:
            self.xms_getter.dataset_uuid = previous_input['elevation_dataset']
            if self.xms_getter.retrieve_data():
                found_previous_dataset = True
                self.data = ManningNData(option, units, in_values, previous_input)
                self.data.set_geometry(self.xms_getter.elevations, self.xms_getter.stations)
            else:
                self.xms_getter.dataset_uuid = ''
        if not found_previous_dataset:  # No previous input or previous input no longer valid
            self.data = ManningNData(option, units, in_values)

        self.exit_h_flow = None
        self.exit_h_hour = None
        self.exit_h_wse = None

        self.ui = Ui_ManningNDialog()
        self.ui.setupUi(self)
        self.setMinimumSize(self.size())

        self.station_header = "Station "
        self.elevation_header = "Elevation "

        self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Channel_Calculator'

        # Map Plot Library
        self.figure = None  # matplotlib.figure Figure
        self.canvas = None  # matplotlib.backends.backend_qt5agg FigureCanvas
        self.axes = None  # matplotlib Axes
        self.initialize_plot()
        self.ui.lay_plot.addWidget(self.canvas)

        self.setup_controls()
        self.setup_connections()
        # format the table
        self.compute_data()

    def add_splitter(self):
        """Adds a QSplitter between the table and plot so the sizes can be adjusted."""
        # The only way this seems to work right is to parent it to
        # self and then insert it into the layout.
        splitter = QSplitter(self)
        splitter.setOrientation(Qt.Horizontal)
        splitter.addWidget(self.ui.lay_table)
        splitter.addWidget(self.canvas)
        # Just use a fixed starting width of 300 for the table for now
        splitter.setSizes([200, 400])
        splitter.setChildrenCollapsible(False)
        splitter.setStyleSheet(
            'QSplitter::handle:horizontal { background-color: lightgrey; }'
            'QSplitter::handle:vertical { background-color: lightgrey; }'
        )
        pos = self.ui.lay_results.indexOf(self.canvas)
        self.ui.lay_results.insertWidget(pos, splitter)

    def setup_controls(self):
        """Set up the controls for the dialog."""
        if self.data.depth_type == "Normal depth":
            self.ui.cbx_type.setCurrentIndex(0)
        else:
            self.ui.cbx_type.setCurrentIndex(1)

        # if self.data.units == "U.S. Customary Units":
        #     self.ui.cbx_units.setCurrentIndex(0)
        # else:
        #     self.ui.cbx_units.setCurrentIndex(1)
        self.update_unit_txt()

        # Composite n
        high_precision = 4
        dbl_valid = QxDoubleValidator(bottom=0., top=1., decimals=high_precision, parent=self)
        self.ui.edt_manning_n.setValidator(dbl_valid)
        initial_string = str(self.data.manning_n_calc.composite_n)
        self.ui.edt_manning_n.setText(initial_string)

        # Slope
        highest_precision = 6
        dbl_valid = QxDoubleValidator(bottom=0., top=1., decimals=highest_precision, parent=self)
        self.ui.edt_slope.setValidator(dbl_valid)
        initial_string = str(self.data.manning_n_calc.slope)
        self.ui.edt_slope.setText(initial_string)

        # Flows
        low_precision = 2
        dbl_valid = QxDoubleValidator(bottom=0., top=10000000.0, decimals=low_precision, parent=self)
        self.ui.edt_flow.setValidator(dbl_valid)
        flow_str = "0.0"
        if len(self.data.manning_n_calc.flows) > 0:
            flow_str = str(self.data.manning_n_calc.flows[0])
        initial_string = flow_str
        self.ui.edt_flow.setText(initial_string)

        self.ui.edt_min_flow.setValidator(dbl_valid)
        initial_string = str(0.0)
        self.ui.edt_min_flow.setText(initial_string)

        self.ui.edt_max_flow.setValidator(dbl_valid)
        initial_string = str(0.0)
        self.ui.edt_max_flow.setText(initial_string)

        self.ui.edt_increment_flow.setValidator(dbl_valid)
        initial_string = str(0.0)
        self.ui.edt_increment_flow.setText(initial_string)

        if self.data.use_multiple_flows:
            self.ui.lay_flow.hide()
        else:
            self.ui.grp_multiple_discharges.hide()
            self.ui.lay_table_btns.hide()

        # Free Board
        self.ui.edt_freeboard.setValidator(dbl_valid)
        initial_string = str(self.data.freeboard)
        self.ui.edt_freeboard.setText(initial_string)

        # Selected dataset name
        if self.xms_getter.dataset_name:
            self.ui.txt_elev_selected.setText(self.xms_getter.dataset_name)

        widget_builder.style_table_view(self.ui.table_data)

    def setup_connections(self):
        """Setup signals and slots for the dialog's controls."""
        self.ui.btn_help.clicked.connect(self.help_requested)

        self.ui.btn_elev.clicked.connect(self.select_dataset)

        self.ui.cbx_type.currentIndexChanged.connect(self.compute_data)
        self.ui.edt_manning_n.textEdited.connect(self.compute_data)
        self.ui.edt_slope.textEdited.connect(self.compute_data)
        self.ui.edt_flow.textEdited.connect(self.compute_data)
        self.ui.edt_freeboard.textEdited.connect(self.compute_data)

        self.ui.btn_add.clicked.connect(self.add_flows)

        self.ui.btn_insert.clicked.connect(self.add_flow)
        self.ui.btn_delete.clicked.connect(self.remove_flow)

        add_icon = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'resources', 'icons', 'add.svg')
        self.ui.btn_insert.icon = QtGui.QIcon(add_icon)
        self.ui.btn_delete.icon = QtGui.QIcon(':/resources/icons/delete.svg')

        self.ui.table_data.cellChanged.connect(self.edit_flow)
        self.ui.table_data.itemSelectionChanged.connect(self.setup_plot)
        self.ui.table_data.installEventFilter(self)

    def eventFilter(self, source, event):  # noqa: N802
        """Handle copy and paste events.

        Args:
            source (:obj:`QObject`): Qt object to handle event for
            event (:obj:`QEvent`): The Qt event to handle

        Returns:
            (:obj:`bool`): True if the event was handled
        """
        if event.type() == QtCore.QEvent.KeyPress and event.matches(QtGui.QKeySequence.Copy):
            self.copy_selection()
            return True
        if event.type() == QtCore.QEvent.KeyPress and event.matches(QtGui.QKeySequence.Paste):
            self.paste_selection()
            return True
        return super(ManningNDialog, self).eventFilter(source, event)

    def copy_selection(self):
        """Handle copy events for the flows table."""
        selection = self.ui.table_data.selectedIndexes()
        if selection:
            rows = sorted(index.row() for index in selection)
            columns = sorted(index.column() for index in selection)
            row_count = rows[-1] - rows[0] + 1
            col_count = columns[-1] - columns[0] + 1
            table = [[''] * col_count for _ in range(row_count)]
            for index in selection:
                row = index.row() - rows[0]
                column = index.column() - columns[0]
                table[row][column] = index.data()
            stream = io.StringIO()
            csv.writer(stream, delimiter='\t').writerows(table)
            QApplication.clipboard().setText(stream.getvalue())

    def paste_selection(self):
        """Handle paste events for the flows table."""
        starting_row_index = 0
        starting_col_index = 0
        # determine our starting point, if the user selected a cell
        selection = self.ui.table_data.selectedIndexes()
        if selection:
            rows = sorted(index.row() for index in selection)
            columns = sorted(index.column() for index in selection)
            starting_row_index = rows[0]
            starting_col_index = columns[0]

        # Pull the data off the clipboard and put it in an array
        buffer = QApplication.clipboard().text()
        reader = csv.reader(io.StringIO(buffer), delimiter='\t')
        array = [[cell for cell in row] for row in reader]
        # Ad rows if the clipboard paste will go beyond the current rows
        if starting_row_index + len(array) > self.ui.table_data.rowCount():
            self.ui.table_data.setRowCount(starting_row_index + len(array))
        # Put the data in the table
        row_index = starting_row_index
        i = 0
        self.ui.table_data.blockSignals(True)
        model = self.ui.table_data.model()
        for line in array:
            col_index = starting_col_index
            j = 0
            for cell in line:
                model.setData(model.index(row_index, col_index), cell)
                col_index += 1
                j += 1
            row_index += 1
            i += 1

        self._set_data_from_table()
        return

    def _set_data_from_table(self):
        """Compute data from current values in the flows table."""
        cur_col = 0
        model = self.ui.table_data.model()
        num_rows = self.ui.table_data.rowCount()

        hours = [0.0] * num_rows
        if self.data.option == "Time series":
            self.data.hours = []
            for i in range(0, num_rows):
                index = model.index(i, cur_col)
                hours[i] = float(model.data(index))
            cur_col += 1

        flows = [0.0] * num_rows
        for i in range(0, num_rows):
            index = model.index(i, cur_col)
            flows[i] = float(model.data(index))

        self.data.set_flows(flows, hours)
        self.compute_data()

    def select_dataset(self):
        """Launches the select dataset dialog and sets the cross section values."""
        self.xms_getter.check_geometry_node_id_gaps = True
        if self.xms_getter.select_dataset(self, True):
            self.data.set_geometry(self.xms_getter.elevations, self.xms_getter.stations)
            self.ui.txt_elev_selected.setText(self.xms_getter.dataset_name)
            self.compute_data()
        self.xms_getter.check_geometry_node_id_gaps = False

    def update_unit_txt(self):
        """Update the unit labels when the units are changed."""
        if self.data.units == "SI Units (Metric)":
            self.ui.txt_elev.setText("Ground elevation dataset (m):")
            self.ui.txt_slope_units.setText("m/m")
            self.ui.txt_min_flow_units.setText("cms")
            self.ui.txt_max_flow_units.setText("cms")
            self.ui.txt_inc_flow_units.setText("cms")
            self.ui.txt_flow_units.setText("cms")
            self.ui.txt_freeboard_units.setText("m")
        else:
            self.ui.txt_elev.setText("Ground elevation dataset (ft):")
            self.ui.txt_slope_units.setText("ft/ft")
            self.ui.txt_min_flow_units.setText("cfs")
            self.ui.txt_max_flow_units.setText("cfs")
            self.ui.txt_inc_flow_units.setText("cfs")
            self.ui.txt_flow_units.setText("cfs")
            self.ui.txt_freeboard_units.setText("ft")

    def setup_manning_n_calc_data(self):
        """Get the data from the dialog, convert, and set to the Manning n calculator class."""
        input_dict = self.get_dialog_data_dict()
        self.data.set_calc_data(
            input_dict['depth_type'], input_dict['manning_n'], input_dict['slope'], input_dict['flow'],
            input_dict['freeboard']
        )

        if not self.data.use_multiple_flows:
            self.ui.table_data.setRowCount(self.data.num_rows)

    def setup_table_headers(self):
        """Setup the headers for the table, complete with proper, selected units."""
        time_header = "Hours"
        flow_header = "Flow "
        self.station_header = "Station "
        self.elevation_header = "Elevation "
        wse_header = "WSE "
        freeboard_header = "WSE offset "
        if self.data.units == "U.S. Customary Units":
            flow_header += "(cfs)"
            self.station_header += "(ft)"
            self.elevation_header += "(ft)"
            wse_header += "(ft)"
            freeboard_header += "(ft)"
        else:
            flow_header += "(cms)"
            self.station_header += "(m)"
            self.elevation_header += "(m)"
            wse_header += "(m)"
            freeboard_header += "(m)"
        headers = []
        if self.data.option == 'Time series':
            headers = [time_header]
        headers.append(flow_header)
        headers.append(wse_header)

        if self.data.freeboard > 0.0:
            headers.append(freeboard_header)
        self.ui.table_data.setHorizontalHeaderLabels(headers)

    def compute_data(self):
        """Run the Manning n Calculator and display the results."""
        self.ui.table_data.blockSignals(True)
        self.data.num_rows = 0
        self.setup_manning_n_calc_data()

        # setup rows and columns and headers
        self.ui.table_data.setRowCount(self.data.num_rows)
        num_cols = 2
        if self.data.freeboard > 0.0:
            num_cols += 1
        if self.data.option == 'Time series':
            num_cols += 1
        self.ui.table_data.setColumnCount(num_cols)
        self.setup_table_headers()

        # Perform the computations and put the results in the table
        success, warnings = self.data.compute_data()
        self.ui.buttonBox.button(QDialogButtonBox.Ok).setEnabled(success)

        self.ui.table_data.setRowCount(self.data.num_rows)
        self.update_table_data()

        self.ui.table_data.resizeColumnsToContents()
        # self.ui.table_data.horizontalHeader().setStretchLastSection(True)
        self.ui.table_data.blockSignals(False)

        self.setup_warning_table(warnings)
        self.setup_plot()
        return

    def update_table_data(self):
        """Adds the list of WSE to the table (also setting up the hour, flow, and freeboard columns."""
        self.exit_h_hour = []
        self.exit_h_flow = []
        self.exit_h_wse = []
        results = self.data.exit_h_results
        col_index = 0
        if 'hrs' in results:
            for index, hour in enumerate(results['hrs']):
                self.ui.table_data.setItem(index, col_index, QTableWidgetItem(f"{hour:.2f}"))
                self.exit_h_hour.append(hour)
            col_index += 1

        for index, flow in enumerate(results['flow']):
            converted_flow_value = flow
            item = QTableWidgetItem(f"{converted_flow_value:.3f}")
            self.exit_h_flow.append(flow)
            if self.data.option == 'Constant':
                item.setFlags(~QtCore.Qt.ItemIsEditable)
            self.ui.table_data.setItem(index, col_index, item)
        col_index += 1

        for index, wse in enumerate(results['wse']):
            if wse == self.data.no_data:
                item = QTableWidgetItem("")
            else:
                item = QTableWidgetItem(f"{wse:.3f}")
            self.exit_h_wse.append(wse)
            item.setFlags(~QtCore.Qt.ItemIsEditable)
            self.ui.table_data.setItem(index, col_index, item)
        col_index += 1

        if 'wse_freeboard' in results:
            for index, wse in enumerate(results['wse_freeboard']):
                if wse == self.data.no_data:
                    item = QTableWidgetItem("")
                else:
                    item = QTableWidgetItem(f"{wse:.3f}")
                self.exit_h_wse[index] = wse
                item.setFlags(~QtCore.Qt.ItemIsEditable)
                self.ui.table_data.setItem(index, col_index, item)
            col_index += 1

    def setup_warning_table(self, warnings):
        """Sets up the warning table.

        Args:
            warnings (:obj:`list[str]`): warnings from the calculations
        """
        self.ui.txt_warnings.clear()
        index = 0
        for warning in warnings:
            # add warning to the table
            self.ui.txt_warnings.append(warning)
            index += 1
        if index == 0:
            self.ui.txt_warnings.append("Computations completed with no warnings")

    def add_flows(self):
        """Populate the flows table using user-specified flow minimum, maximum, and increment."""
        min_flow = float(self.ui.edt_min_flow.text())
        max_flow = float(self.ui.edt_max_flow.text())
        inc_flow = float(self.ui.edt_increment_flow.text())
        app_name = os.environ.get('XMS_PYTHON_APP_NAME')
        if min_flow > max_flow:
            msg = '"Minimum flow" must be less than "Maximum flow".'
            message_with_ok(self, msg, app_name)
            return
        if inc_flow == 0.0:
            msg = 'The "Increment of flow" must be greater than zero.'
            message_with_ok(self, msg, app_name)
            return
        delta_flow = max_flow - min_flow
        num_flows = delta_flow / inc_flow
        if num_flows > 50:
            msg = 'The number of flows that will be added to the table is greater than 50. ' \
                  'This computation may take several minutes to compute. Continue?'
            ret = message_with_n_buttons(self, msg, app_name, ['Yes', 'No'], 1, 1)
            if ret == 1:
                return
        self.data.add_flows(min_flow, max_flow, inc_flow)
        self.compute_data()

    def add_flow(self):
        """Add a row to the flows table."""
        self.data.add_flow()
        self.compute_data()

    def remove_flow(self):
        """Remove the currently selected row from the flows table."""
        selected_row = self.ui.table_data.currentRow()
        self.data.remove_flow(selected_row)
        self.compute_data()

    def edit_flow(self, row, col):
        """Edit an item in the flows table.

        Args:
            row (:obj:`int`): Index of the row to edit
            col (:obj:`int`): Index of the column to edit
        """
        if row < len(self.data.flows):
            if self.data.option == 'Time series' and col == 0:
                value = float(self.ui.table_data.item(row, col).text())
            else:
                value = float(self.ui.table_data.item(row, col).text())
            self.data.set_flow(row, col, value)
        self.compute_data()

    def initialize_plot(self):
        """Sets up the plot."""
        self.figure = Figure(tight_layout=True)
        self.canvas = FigureCanvas(self.figure)
        # self.ui.plt_channel = FigureCanvas(self.figure)
        self.setup_plot()

    def setup_plot(self):
        """Add a new plot to the data model and draw it."""
        if not self.axes:
            self.axes = self.figure.subplots()
        selected_row = self.ui.table_data.currentRow()
        self.data.add_series(self.axes, selected_row, self.station_header, self.elevation_header)
        self.canvas.draw()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def get_dialog_data_dict(self):
        """Returns a dict of the dialog input field widgets."""
        return {
            'depth_type': self.ui.cbx_type.currentText(),
            'elevation_dataset': self.xms_getter.dataset_uuid,
            'geom_uuid': self.xms_getter.mesh_uuid if self.xms_getter.mesh_uuid else '',
            'manning_n': str_to_float(self.ui.edt_manning_n.text(), 0.0),
            'slope': str_to_float(self.ui.edt_slope.text(), 0.0),
            'freeboard': str_to_float(self.ui.edt_freeboard.text(), 0.0),
            'flow': str_to_float(self.ui.edt_flow.text(), 0.0),
        }
