"""Dialog for defining SRH-2D parameter run simulations."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QAbstractItemView, QDialog, QSplitter

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.settings import SettingsManager
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.srh.components.parameters.parameter_runs_filter_proxy_model import ParameterRunsFilterProxyModel
from xms.srh.components.sim_query_helper import SimQueryHelper
from xms.srh.gui.parameters_dialog_ui import Ui_dlg_parameters
from xms.srh.gui.parameters_xy_series_dialog import ParametersXySeriesDialog
from xms.srh.mapping.coverage_mapper import CoverageMapper
from xms.srh.model.srh_pest_model import get_computed_values


class ParametersDialog(XmsDlg):
    """A dialog used to define scenarios where parameters can be varied, or to use with PEST.

    Currently this code is very SRH-2D specific. It could be made more generic but we decided to postpone that
    refactoring effort until we actually had another model to apply it to. In the meantime, I've written these
    comments so that future programmers would have a clue.

    There are two tables: "Available parameters" (params) and "Model runs and values" (runs). Each parameter in the
    params table is a class that has the responsibility of adding columns and delegates to the runs table. A single
    parameter can result in multiple columns in the runs table.

    Michael Kennard
    """

    # Columns in the params table
    PARAMS_ID_COLUMN = 0
    PARAMS_USE_COLUMN = 1
    PARAMS_TYPE_COLUMN = 2
    PARAMS_DESCRIPTION_COLUMN = 3
    PARAMS_DEFAULT_COLUMN = 4
    PARAMS_STRING_VALUE_COLUMN = 5
    PARAMS_VALUE_COLUMN = 6
    PARAMS_OPTIMIZED_VALUE_COLUMN = 7
    PARAMS_MIN_COLUMN = 8
    PARAMS_MAX_COLUMN = 9

    # Columns in the monitor points table
    MONITORS_ID_COLUMN = 0
    MONITORS_MEASURED_COLUMN = 1
    MONITORS_COMPUTED_COLUMN = 2
    MONITORS_DIFFERENCE_COLUMN = 3

    # Columns in the runs table (only the first as the others are dynamic)
    RUNS_RUN_COLUMN = 0

    # Run count constants
    RUN_COUNT_MIN = 1
    RUN_COUNT_MAX = 10000

    def __init__(
        self,
        param_data,
        pe_tree_trimmed,
        parent=None,
        query=None,
        srh_dir=None,
        case_name=None,
        monitor_pts=None,
        sim_helper=None
    ):
        """Initializes the class, sets up the ui.

        Args:
            param_data (:obj:`ParameterData`): Object with all the info the dialog needs.
            pe_tree_trimmed (:obj:`xms.guipy.tree.tree_node.TreeNode`): The XMS project explorer tree,
                trimmed to the model grid.
            parent (Something derived from :obj:`QWidget`): The parent window.
            query (:obj:`Query`): XMS interprocess communication object
            srh_dir (:obj:`str`): The SRH model directory (used for testing only)
            case_name (:obj:`str`): The SRH model case name (used for testing only)
            monitor_pts (:obj:`list[tuple]`): List containing all the necessary monitor point information
            sim_helper (:obj:`SimQueryHelper`): Query helper object
        """
        super().__init__(parent, 'xms.srh.gui.parameters_dialog')

        self.ui = Ui_dlg_parameters()
        self.ui.setupUi(self)
        self.vert_splitter = None
        self.horiz_splitter = None

        self.param_data = param_data
        self.pe_tree_trimmed = pe_tree_trimmed
        self.params_model = None
        self.monitors_model = None
        self.runs_model = None
        self.xy_series_names = []
        self.filter_model = None
        self.query = query
        self.sim_query_helper = sim_helper
        if not self.sim_query_helper:  # Create a dummy for testing
            self.sim_query_helper = SimQueryHelper(self.query)
        self.coverage_mapper = CoverageMapper(self.sim_query_helper, generate_snap=False)
        self.coverage_mapper.map_monitor_for_advanced_sim_dlg()
        self.sim_item = None
        if query:
            self.sim_item = tree_util.find_tree_node_by_uuid(query.project_tree, self.sim_query_helper.sim_uuid)
        self.monitor_data = self._get_monitor_data(srh_dir, case_name, monitor_pts)

        self.add_accessible_names_for_tests()
        self.add_splitters()
        self.setup_run_type()
        self.setup_filter()
        self.setup_params_table()
        self.setup_monitors_table()
        self.setup_run_count()
        self.setup_pest_data()
        self.setup_runs_table()
        self.setup_signals()
        self.setup_context_menus()
        self.ui.grp_use_parameters.setChecked(self.param_data.use_parameters)
        self.on_run_type()
        self.on_show_type()
        self.on_use_parameters()

    def add_accessible_names_for_tests(self):
        """Adds accessible names for some widgets so that they can be found by GuiTester."""
        self.ui.cbx_run_type.setAccessibleName('Run type combo box')
        self.ui.cbx_show_type.setAccessibleName('Show type combo box')
        self.ui.wid_left.setAccessibleName('Left widget')
        self.ui.wid_left_top.setAccessibleName('Left Top widget')
        self.ui.wid_right.setAccessibleName('Right widget')
        self.ui.tbl_parameters.setAccessibleName('Parameters table')
        self.ui.tbl_monitors.setAccessibleName('Monitor points table')
        self.ui.spn_runs.setAccessibleName('Runs spin control')
        self.ui.tbl_runs.setAccessibleName('Runs table')

    def _get_monitor_data(self, srh_dir=None, case_name=None, monitor_pts=None):
        """Returns the monitor point data dict from the coverage mapper."""
        data = {'Monitor Point ID': [], 'Measured Value': [], 'Computed Value': [], 'Difference': []}
        project_file = os.getenv('XMS_PYTHON_APP_PROJECT_PATH')
        if self.coverage_mapper.monitor_points:
            monitor_pts = self.coverage_mapper.monitor_points
        if (monitor_pts and project_file) or srh_dir:
            if project_file:
                project_dir = os.path.dirname(project_file)
                project_name = os.path.basename(os.path.splitext(project_file)[0])
                srh_dir = os.path.join(project_dir, f'{project_name}_models', 'SRH-2D', self.sim_item.name)
            if self.sim_query_helper.sim_component:
                case_name = self.sim_query_helper.sim_component.data.hydro.case_name
            model_data = {
                'PROJECT_NAME': "",
                'CASE_NAME': case_name,
                'OBSERVATIONS': [],
                'MONITOR_PTS': [pt[3] for pt in monitor_pts],
                'PARAMETER_NAMES': [],
            }
            data['Computed Value'] = get_computed_values(model_data, folder=srh_dir)
            for pt in monitor_pts:
                if pt[3] is True:
                    data['Monitor Point ID'].append(pt[0])
                    data['Measured Value'].append(pt[4])
                    idx = len(data['Measured Value']) - 1
                    data['Difference'].append(data['Computed Value'][idx] - data['Measured Value'][idx])
        return data

    def _save_splitter_geometry(self):
        """Save the current position of the splitter."""
        settings = SettingsManager()
        settings.save_setting('xmssrh', f'{self._dlg_name}.splitter', self.vert_splitter.sizes())
        settings.save_setting('xmssrh', f'{self._dlg_name}.horiz_splitter', self.horiz_splitter.sizes())

    def _restore_splitter_geometry(self):
        """Restore the position of the splitter."""
        settings = SettingsManager()
        splitter = settings.get_setting('xmssrh', f'{self._dlg_name}.splitter')
        if not splitter:
            return
        splitter_sizes = [int(size) for size in splitter]
        self.vert_splitter.setSizes(splitter_sizes)
        splitter = settings.get_setting('xmssrh', f'{self._dlg_name}.horiz_splitter')
        if not splitter:
            return
        splitter_sizes = [int(size) for size in splitter]
        self.horiz_splitter.setSizes(splitter_sizes)

    def _check_parameter_values(self):
        """Called during accept to make sure that the parameter value is between the min/max.

        Returns:
            (:obj:`bool`): true if the parameter data does not require edits
        """
        msg = 'Parameter values must be between the min/max of the parameter. The following parameters must ' \
              'be edited so that the value is between the min/max:\n\n'
        show_msg = False
        self._update_param_data()
        if self.param_data.use_parameters is True and self.param_data.run_type == 'Calibration':
            for parameter in self.param_data.params:
                if parameter.use == 1:
                    if parameter.value < parameter.min or parameter.value > parameter.max:
                        if show_msg:
                            msg += ', '
                        show_msg = True
                        msg += f'{parameter.type}-{parameter.description}'
        if show_msg:
            app_name = os.environ.get('XMS_PYTHON_APP_NAME')
            message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())

            return False
        return True

    def accept(self):
        """Save position and geometry when closing dialog."""
        if not self._check_parameter_values():
            return
        self._save_splitter_geometry()
        super().accept()

    def reject(self):
        """Save position and geometry when closing dialog."""
        self._save_splitter_geometry()
        super().reject()

    def showEvent(self, event):  # noqa: N802
        """Restore last position and geometry when showing dialog."""
        super().showEvent(event)
        self._restore_splitter_geometry()

    def add_splitters(self):
        """Adds a QSplitter between the tables so the sizes can be adjusted."""
        # The only way this seems to work right is to parent it to
        # self and then insert it into the layout.
        # Add vertical splitter
        self.vert_splitter = QSplitter(self)
        self.vert_splitter.setOrientation(Qt.Horizontal)
        self.vert_splitter.addWidget(self.ui.wid_left)
        self.vert_splitter.addWidget(self.ui.wid_right)
        self.vert_splitter.setSizes([400, 600])
        self.vert_splitter.setChildrenCollapsible(False)
        self.vert_splitter.setStyleSheet(
            'QSplitter::handle:horizontal { background-color: lightgrey; }'
            'QSplitter::handle:vertical { background-color: lightgrey; }'
        )
        self.vert_splitter.setAccessibleName('Vertical Splitter')
        pos = self.ui.hlay_group_box.indexOf(self.ui.wid_right)
        self.ui.hlay_group_box.insertWidget(pos, self.vert_splitter)

        # Add horizontal splitter
        self.horiz_splitter = QSplitter(self)
        self.horiz_splitter.setOrientation(Qt.Vertical)
        self.horiz_splitter.addWidget(self.ui.wid_left_top)
        self.horiz_splitter.addWidget(self.ui.wid_left_bot)
        self.horiz_splitter.setSizes([800, 200])
        self.horiz_splitter.setChildrenCollapsible(False)
        self.horiz_splitter.setStyleSheet(
            'QSplitter::handle:horizontal { background-color: lightgrey; }'
            'QSplitter::handle:vertical { background-color: lightgrey; }'
        )
        self.horiz_splitter.setAccessibleName('Horizontal Splitter')
        pos = self.ui.vlay_wid_left.indexOf(self.ui.wid_left_bot)
        self.ui.vlay_wid_left.insertWidget(pos, self.horiz_splitter)

    def setup_run_type(self):
        """Initialize the run type combobox."""
        self.ui.cbx_run_type.addItems(['Scenarios', 'Calibration'])
        self.ui.cbx_run_type.setCurrentText(self.param_data.run_type)

    def setup_filter(self):
        """Initialize the filter combobox."""
        self.ui.cbx_show_type.addItems(
            ['All', 'Inlet Q', 'Exit H', 'Internal sink', "Manning's N", 'Time step', 'Initial condition']
        )
        self.ui.cbx_show_type.setCurrentText(self.param_data.show_type)

    def setup_signals(self):
        """Sets up the signal."""
        self.ui.grp_use_parameters.clicked.connect(self.on_use_parameters)
        self.ui.cbx_run_type.currentIndexChanged.connect(self.on_run_type)
        self.ui.cbx_show_type.currentIndexChanged.connect(self.on_show_type)
        self.params_model.dataChanged.connect(self.on_params_change)
        self.ui.spn_runs.valueChanged.connect(self.on_run_count_changed)
        self.ui.btn_xy_series.clicked.connect(self.on_xy_series)

    def setup_context_menus(self):
        """Sets up the context menus."""
        self.ui.tbl_runs.verticalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
        self.ui.tbl_runs.verticalHeader().customContextMenuRequested.connect(self.on_index_column_click)
        self.ui.tbl_runs.setContextMenuPolicy(Qt.CustomContextMenu)
        self.ui.tbl_runs.customContextMenuRequested.connect(self.on_right_click)

    def on_right_click(self, point):
        """Slot called when user right-clicks in the table.

        Args:
            point(:obj:`QPoint`): The point clicked.
        """
        # row = self.ui.table_view.logicalIndexAt(point)
        menu_list = [['', 'Copy', self.ui.tbl_runs.on_copy], ['', 'Paste', self.ui.tbl_runs.on_paste]]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.ui.tbl_runs.viewport().mapToGlobal(point))

    def on_index_column_click(self, point):
        """Called on right-click on the index column (vertical header).

        Args:
            point (:obj:`QPoint`): The point clicked
        """
        row = self.ui.tbl_runs.verticalHeader().logicalIndexAt(point)
        self.ui.tbl_runs.selectRow(row)
        menu_list = [['', 'Copy', self.ui.tbl_runs.on_copy], ['', 'Paste', self.ui.tbl_runs.on_paste]]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.ui.tbl_runs.viewport().mapToGlobal(point))

    def on_xy_series(self):
        """Called when XY Series button is clicked."""
        dialog = ParametersXySeriesDialog(xy_series_list=self.param_data.xy_series, parent=self)
        if dialog.exec() == QDialog.Accepted:
            self.param_data.xy_series = dialog.get_xy_series_data()
            self.add_combo_boxes_to_runs_model()

    def on_run_count_changed(self, value):
        """Called when 'Number of runs' spin control changes.

        Args:
            value (:obj:`int`): New value in the spin control.
        """
        old_row_count = self.runs_model.rowCount()
        new_row_count = value
        if new_row_count > old_row_count:
            self.runs_model.insertRows(old_row_count, new_row_count - old_row_count)
        else:
            self.runs_model.removeRows(new_row_count, old_row_count - new_row_count)

        # Update Runs column
        if new_row_count > old_row_count:
            for row in range(old_row_count, new_row_count):
                index = self.runs_model.createIndex(row, self.RUNS_RUN_COLUMN)
                self.runs_model.setData(index, f'Run {row + 1}')
            self.runs_model.submit()

    def on_show_type(self):
        """Called when the 'Show type' combo box is changed."""
        showing = self.ui.cbx_show_type.currentText()
        for row in range(self.params_model.rowCount()):
            if showing == 'All':
                self.ui.tbl_parameters.setRowHidden(row, False)
            else:
                index = self.params_model.createIndex(row, self.PARAMS_TYPE_COLUMN)
                type = self.params_model.data(index)
                self.ui.tbl_parameters.setRowHidden(row, type != showing)

    def on_run_type(self):
        """Called when the 'Run type' combo box is changed."""
        hide_pest_data = self.ui.cbx_run_type.currentText() == 'Scenarios'
        self.ui.wid_pest_parameters.setHidden(hide_pest_data)
        self.ui.txt_calibration_notes.setHidden(hide_pest_data)
        self.ui.wid_right.setVisible(hide_pest_data)
        self.show_hide_pest_data()

    def on_use_parameters(self):
        """Called when the 'Use parameters' toggle is clicked."""
        checked = self.ui.grp_use_parameters.isChecked()

        self.ui.txt_run_type.setEnabled(checked)
        self.ui.cbx_run_type.setEnabled(checked)
        self.ui.txt_show_type.setEnabled(checked)
        self.ui.cbx_show_type.setEnabled(checked)
        self.ui.txt_parameters.setEnabled(checked)
        self.ui.tbl_parameters.setEnabled(checked)
        self.ui.tbl_monitors.setEnabled(checked)

        self.ui.txt_runs.setEnabled(checked)
        self.ui.spn_runs.setEnabled(checked)
        self.ui.btn_xy_series.setEnabled(checked)
        self.ui.tbl_runs.setEnabled(checked)

    def on_params_change(self, top_left, bottom_right):
        """Called when there's a change in the params table (checkbox clicked).

        Args:
            top_left (:obj:`QModelIndex`): Index of the topmost, leftmost item that changed
            bottom_right (:obj:`QModelIndex`): Index of the bottommost, rightmost item that changed
        """
        col_start = top_left.column()
        col_end = bottom_right.column()
        row_start = top_left.row()
        row_end = bottom_right.row()
        self.update_params_table(col_start, col_end, row_start, row_end)

    def update_params_table(self, col_start, col_end, row_start, row_end):
        """Update the parameters table.

        Args:
            col_start (:obj:`int`): Index of the first column to update
            col_end (:obj:`int`): Index of the last column to update
            row_start (:obj:`int`): Index of the first row to update
            row_end (:obj:`int`): Index of the last row to update
        """
        hide_pest_data = self.ui.cbx_run_type.currentText() == 'Scenarios'
        for column in range(col_start, col_end + 1):
            if column == self.PARAMS_USE_COLUMN:
                for row in range(row_start, row_end + 1):
                    index = self.params_model.createIndex(row, self.PARAMS_USE_COLUMN)
                    checked = self.params_model.data(index, Qt.CheckStateRole) == Qt.Checked
                    parameter = self.param_data.params[row]
                    if not hide_pest_data:
                        checked = checked and parameter.calibratable()
                    self.show_or_hide_runs_table_column(parameter, checked)
                    for col_param in [self.PARAMS_VALUE_COLUMN, self.PARAMS_MIN_COLUMN, self.PARAMS_MAX_COLUMN]:
                        if not checked:
                            self.params_model.read_only_cells.add((row, col_param))
                        else:
                            self.params_model.read_only_cells.discard((row, col_param))
                        index = self.params_model.createIndex(row, col_param)
                        self.params_model.dataChanged.emit(index, index)
        pass  # Here so I can set a breakpoint at the end of this method

    def show_or_hide_runs_table_columns(self):
        """Shows or hides the appropriate columns in the runs table."""
        top_left = self.params_model.createIndex(0, self.PARAMS_USE_COLUMN)
        bottom_right = self.params_model.createIndex(self.params_model.rowCount() - 1, self.PARAMS_USE_COLUMN)
        self.on_params_change(top_left, bottom_right)

    def show_or_hide_runs_table_column(self, parameter, checked):
        """Updates the runs table by adding or removing a column.

        Args:
            parameter (:obj:`Parameter`): The parameter.
            checked (:obj:`bool`): Whether the row is checked or not.
        """
        if not self.runs_model:
            return

        # Count number of columns before the current parameter
        column_offset = 1  # Start at 1 for the Runs column
        for current_parameter in self.param_data.params:
            if current_parameter != parameter:
                column_offset += current_parameter.column_count()
            else:
                break

        # Hide all the columns pertaining to this parameter
        hidden = not checked
        for column in range(parameter.column_count()):
            self.ui.tbl_runs.setColumnHidden(column_offset + column, hidden)

    def setup_params_table(self):
        """Sets up the parameters table."""
        df = pd.DataFrame(self.param_data.dict_from_parameter_list())
        self.params_model = QxPandasTableModel(df)
        self.params_model.set_checkbox_columns({self.PARAMS_USE_COLUMN})
        self.params_model.read_only_columns = {
            self.PARAMS_TYPE_COLUMN, self.PARAMS_DESCRIPTION_COLUMN, self.PARAMS_OPTIMIZED_VALUE_COLUMN
        }
        self.ui.tbl_parameters.setModel(self.params_model)
        self.ui.tbl_parameters.setColumnHidden(self.PARAMS_ID_COLUMN, True)
        self.ui.tbl_parameters.setColumnHidden(self.PARAMS_DEFAULT_COLUMN, True)
        self.ui.tbl_parameters.setColumnHidden(self.PARAMS_STRING_VALUE_COLUMN, True)
        self.show_hide_pest_data()
        widget_builder.style_table_view(self.ui.tbl_parameters)
        self.ui.tbl_parameters.resizeColumnsToContents()
        self.ui.tbl_parameters.verticalHeader().setVisible(False)
        self.ui.tbl_parameters.horizontalHeader().setStretchLastSection(True)

    def setup_monitors_table(self):
        """Sets up the monitor points table."""
        df = pd.DataFrame(self.monitor_data)
        self.monitors_model = QxPandasTableModel(df)
        self.monitors_model.read_only_columns = {i for i in range(self.MONITORS_DIFFERENCE_COLUMN + 1)}
        self.ui.tbl_monitors.setModel(self.monitors_model)
        widget_builder.style_table_view(self.ui.tbl_monitors)
        self.ui.tbl_monitors.resizeColumnsToContents()
        self.ui.tbl_monitors.verticalHeader().setVisible(False)
        self.ui.tbl_monitors.horizontalHeader().setStretchLastSection(True)

    def show_hide_pest_data(self):
        """Shows or hides the pest data in the parameters table."""
        hide_pest_data = self.ui.cbx_run_type.currentText() == 'Scenarios'
        for column in [
            self.PARAMS_VALUE_COLUMN, self.PARAMS_OPTIMIZED_VALUE_COLUMN, self.PARAMS_MIN_COLUMN, self.PARAMS_MAX_COLUMN
        ]:
            self.ui.tbl_parameters.setColumnHidden(column, hide_pest_data)
        for row in range(self.params_model.rowCount()):
            parameter = self.param_data.params[row]
            if hide_pest_data or parameter.calibratable():
                self.params_model.read_only_cells.discard((row, self.PARAMS_USE_COLUMN))
            else:
                self.params_model.read_only_cells.add((row, self.PARAMS_USE_COLUMN))
            index = self.params_model.createIndex(row, self.PARAMS_USE_COLUMN)
            self.params_model.dataChanged.emit(index, index)
        self.update_params_table(self.PARAMS_USE_COLUMN, self.PARAMS_USE_COLUMN, 0, self.params_model.rowCount() - 1)

    def setup_run_count(self):
        """Sets up the run count spin control."""
        # Set runs spin control
        self.ui.spn_runs.setValue(self.param_data.run_count)
        self.ui.spn_runs.setMinimum(self.RUN_COUNT_MIN)
        self.ui.spn_runs.setMaximum(self.RUN_COUNT_MAX)  # Don't want the table to be limitless

    def setup_pest_data(self):
        """Sets up the pest data spin controls."""
        # Set pest data spin controls
        self.ui.spn_max_iterations.setValue(self.param_data.max_iterations)
        self.ui.spn_max_iterations.setMinimum(self.RUN_COUNT_MIN)
        self.ui.spn_max_iterations.setMaximum(self.RUN_COUNT_MAX)
        self.ui.spn_max_iterations_no_improvement.setValue(self.param_data.max_iterations_no_improvement)
        self.ui.spn_max_iterations_no_improvement.setMinimum(self.RUN_COUNT_MIN)
        self.ui.spn_max_iterations_no_improvement.setMaximum(self.RUN_COUNT_MAX)

    def set_run_column_types(self, df):
        """Changes the dataframe column types so that strings can be entered in some.

        Args:
            df (:obj:`Pandas.DataFrame`): The DataFrame to edit
        """
        for column_name in df.columns.values:
            dtype = object
            if column_name != 'Run':
                parameter = self.param_data.parameter_from_column_name(column_name)
                dtype = parameter.column_dtype(column_name)
            if dtype:
                df[column_name] = df[column_name].astype(dtype)

    def add_combo_boxes_to_runs_model(self):
        """Adds combo boxes to the runs table for XY series."""
        # Get list of xy series
        xy_series_names = [series[0] for series in self.param_data.xy_series]
        for col, column_name in enumerate(self.runs_model.data_frame.columns.values):
            if column_name == 'Run':
                continue
            parameter = self.param_data.parameter_from_column_name(column_name)
            delegate = parameter.column_delegate(column_name, xy_series_names, self)
            if delegate:
                self.ui.tbl_runs.setItemDelegateForColumn(col, delegate)

        self.ui.tbl_runs.setEditTriggers(QAbstractItemView.AllEditTriggers)

    def setup_runs_table(self):
        """Sets up the runs table."""
        df = pd.DataFrame(self.param_data.runs)
        df.index += 1
        self.set_run_column_types(df)
        self.runs_model = QxPandasTableModel(df)
        defaults = self.param_data.get_defaults()
        self.runs_model.set_default_values(defaults)
        self.filter_model = ParameterRunsFilterProxyModel(self.param_data, self)
        self.filter_model.setSourceModel(self.runs_model)
        self.ui.tbl_runs.setModel(self.filter_model)
        self.add_combo_boxes_to_runs_model()
        widget_builder.style_table_view(self.ui.tbl_runs)
        self.ui.tbl_runs.resizeColumnsToContents()

        self.show_or_hide_runs_table_columns()

    def _update_param_data(self):
        """Updates param_data with the data from the dialog.
        """
        self.param_data.use_parameters = self.ui.grp_use_parameters.isChecked()
        self.param_data.run_type = self.ui.cbx_run_type.currentText()
        self.param_data.show_type = self.ui.cbx_show_type.currentText()
        self.param_data.max_iterations = self.ui.spn_max_iterations.value()
        self.param_data.max_iterations_no_improvement = self.ui.spn_max_iterations_no_improvement.value()
        self.param_data.run_count = self.ui.spn_runs.value()
        # self.param_data.xy_series = Updated dynamically already

        self._update_parameters_from_model(self.params_model)
        self.param_data.runs = self._runs_dict_from_table(self.runs_model)

    def _update_parameters_from_model(self, params_model):
        """Updates parameters from the model.

        Returns:
            (:obj:`list[Parameter]`): list of Parameters.
        """
        df = params_model.data_frame
        parameters = []
        for row in range(params_model.rowCount()):
            self.param_data.params[row].update(
                use=df.iloc[row]['Use'], value=df.iloc[row]['Value'], min=df.iloc[row]['Min'], max=df.iloc[row]['Max']
            )
        return parameters

    @staticmethod
    def _runs_dict_from_table(model):
        df = model.data_frame
        _dict = {}
        for i in range(model.columnCount()):
            column_name = df.columns.values[i]
            _dict[column_name] = df[df.columns[i]].to_list()
        return _dict
