"""The Structures Map dialog."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWebEngineWidgets import QWebEngineView
from PySide2.QtWidgets import (QDialogButtonBox, QHBoxLayout, QVBoxLayout, QWidget)

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules


class StructuresMapDialog(XmsDlg):
    """A dialog for showing the Structure Plot."""
    def __init__(self, win_cont: QWidget, file_path: str | Path):
        """Initializes the dialog, sets up the ui.

        Args:
            win_cont: Parent dialog.
            file_path: file path to the location of the structures_map.html file.
        """
        super().__init__(win_cont, 'xms.srh.gui.structures_map_dialog')
        self._file_path = str(Path(file_path).as_posix())  # Folium requires forward slashes '/'
        self._structures_plot = None
        self.widgets = {}
        self.setWindowTitle('Structures Map')
        self._setup_ui()

    def _setup_ui(self) -> None:
        """Sets up the dialog controls."""
        self.widgets['main_vert_layout'] = QVBoxLayout()
        self.setLayout(self.widgets['main_vert_layout'])
        self._setup_structures_plot()
        self._setup_ui_bottom_button_box()

    def _setup_structures_plot(self) -> None:
        """Sets up the folium map display."""
        self._structures_plot = QWebEngineView()  # Widget for rendering HTML content
        self._structures_plot.setMinimumWidth(300)  # Prevent the map from being resized too small
        self.widgets['main_vert_layout'].addWidget(self._structures_plot)
        self._structures_plot.setUrl(f'file:///{self._file_path}')

    def _setup_ui_bottom_button_box(self):
        """Add buttons to the bottom of the dialog."""
        # Add Import and Export buttons
        self.widgets['btn_horiz_layout'] = QHBoxLayout()
        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['btn_box'].setOrientation(Qt.Horizontal)
        self.widgets['btn_box'].setStandardButtons(QDialogButtonBox.Close)
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(self.reject)
        self.widgets['btn_horiz_layout'].addWidget(self.widgets['btn_box'])
        self.widgets['main_vert_layout'].addLayout(self.widgets['btn_horiz_layout'])
        self.widgets['btn_box'].setFocus()
