"""Qt delegate for the material polygon fill texture button widget."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, QSize, Qt
from PySide2.QtGui import QBrush, QColor, QPixmap
from PySide2.QtWidgets import QDialog, QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.dialogs.polygon_display_options import PolygonDisplayOptionsDialog
from xms.guipy.widgets.display_option_icon_factory import DisplayOptionIconFactory

# 4. Local modules


class SrhDisplayOptionDelegate(QStyledItemDelegate):
    """Delegate for the material polygon fill texture button."""
    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (:obj:`QWidget`): The editor widget.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (:obj:`QPainter`): The painter.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            # special button with icon
            # remove 4 from icon size so it doesn't overlap button border
            min_dim = min(option.rect.width(), option.rect.height()) - 4
            icon_size = QSize(min_dim, min_dim)
            btn.setIconSize(icon_size)
            btn.setIcon(DisplayOptionIconFactory.get_icon(self.str_to_poly_opts(index.data(Qt.EditRole)), min_dim))
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())

            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (:obj:`QEvent`): The editor event that was triggered.
            model (:obj:`QAbstractItemModel`): The data model.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                # check the option type, show the correct dialog
                dlg = PolygonDisplayOptionsDialog(self.str_to_poly_opts(index.data(Qt.EditRole)), self.parent())
                if dlg and dlg.exec() == QDialog.Accepted:
                    model.setData(index, self.poly_opts_to_str(dlg.get_options()), Qt.EditRole)
                return True
        return super().editorEvent(event, model, option, index)

    @staticmethod
    def str_to_poly_opts(disp_str):
        """Convert the string representation of polygon color and texture to a PolygonOptions object.

        Args:
            disp_str (:obj:`str`): The color/texture formatted as 'R G B texture' where R, G, B, and texture are
                integers.

        Returns:
            (:obj:`PolygonOptions`): Polygon display options represented by disp_str
        """
        disp_data = str(disp_str).split()  # r, g, b, texture
        poly_opts = PolygonOptions()
        poly_opts.color = QColor(int(disp_data[0]), int(disp_data[1]), int(disp_data[2]), 255)
        poly_opts.texture = int(disp_data[3])
        return poly_opts

    @staticmethod
    def poly_opts_to_str(poly_opts):
        """Convert a PolygonOptions object to its string representation of polygon color and texture.

        Args:
            poly_opts (:obj:`PolygonOptions`): Polygon display options to stringify

        Returns:
            (:obj:`str`): The color/texture formatted as 'R G B texture' where R, G, B, and texture are integers.
        """
        return f'{poly_opts.color.red()} {poly_opts.color.green()} {poly_opts.color.blue()} {int(poly_opts.texture)}'
