"""Map locations and attributes of all linked coverages to the SRH-2D domain."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.gdal.rasters import raster_utils as ru

# 4. Local modules
from xms.srh.data.material_data import MaterialData
from xms.srh.mapping.bc_mapper import BcMapper
from xms.srh.mapping.material_mapper import MaterialMapper
from xms.srh.mapping.monitor_mapper import MonitorMapper
from xms.srh.mapping.obstructions_mapper import ObstructionsMapper


class CoverageMapper:
    """Class for mapping coverages to a mesh for SRH-2D."""
    def __init__(self, query_helper, generate_snap):
        """Constructor."""
        super().__init__()
        self._logger = logging.getLogger('xms.srh')
        self._generate_snap = generate_snap

        self._query_helper = query_helper
        self.grid_units = query_helper.grid_units
        self.grid_name = query_helper.grid_name
        self.co_grid = query_helper.co_grid
        self.grid_uuid = query_helper.grid_uuid
        self.grid_wkt = query_helper.grid_wkt
        self.component_folder = query_helper.component_folder
        self.ceiling_file = None
        self.structures_3d_weirs = query_helper.structures_3d_weirs
        self.structures_3d_monitor = query_helper.structures_3d_monitor

        mat_cov = query_helper.coverages.get('Materials', [None, None])
        self.material_coverage = mat_cov[0]
        self.material_component_file = mat_cov[1]
        self.material_component = query_helper.material_component
        self.material_comp_id_to_grid_cell_ids = None
        self.material_names = None
        self.material_comp_ids = None
        self.material_mannings = None
        self.mapped_material_uuid = None
        self.mapped_material_display_uuid = None

        sed_mat_cov = query_helper.coverages.get('Sediment Materials', [None, None])
        self.sed_material_coverage = sed_mat_cov[0]
        self.sed_material_component_file = sed_mat_cov[1]
        self.sed_material_component = query_helper.sed_material_component
        self.sed_material_comp_id_to_grid_cell_ids = None
        self.sed_material_names = None
        self.sed_material_comp_ids = None
        self.sed_material_data = None
        self.mapped_sed_material_uuid = None
        self.mapped_sed_material_display_uuid = None

        bc_cov = query_helper.coverages.get('Boundary Conditions', [None, None])
        self.bc_coverage = bc_cov[0]
        self.bc_component_file = bc_cov[1]
        self.bc_component = query_helper.bc_component
        self.bc_arc_id_to_grid_ids = None
        self.bc_arc_id_to_grid_pts = None
        self.bc_arc_id_to_node_string_lengths = None
        self.bc_arc_id_to_comp_id = None
        self.bc_arc_id_to_bc_param = None
        self.bc_arc_id_to_bc_id = None
        self.bc_id_to_structures = None
        self.bc_hy8_file = None
        self.bc_3d_structures = []
        self.bc_mapped_comp_uuid = None
        self.bc_mapped_comp_display_uuid = None

        monitor_cov = query_helper.coverages.get('Monitor', [None, None])
        self.monitor_coverage = monitor_cov[0]
        self.monitor_component = query_helper.monitor_component
        self.monitor_points = None
        self.monitor_arc_ids_to_grid_cell_ids = None
        self.monitor_mapped_comp_uuid = None
        self.monitor_mapped_comp_display_uuid = None

        obstruction_cov = query_helper.coverages.get('Obstructions', [None, None])
        self.obstructions_coverage = obstruction_cov[0]
        self.obstructions_component_file = obstruction_cov[1]
        self.obstructions_component = query_helper.obstruction_component
        self.obstructions_decks = None
        self.obstructions_piers = None

        self.mapped_comps = []

    def do_map(self):
        """Creates the snap preview of coverages onto the mesh."""
        if self.co_grid is None:
            self._logger.error('No mesh (or UGrid) linked to the simulation. Aborting.')
            return
        try:
            self._map_materials()
            self._map_sed_materials()
            self._map_monitor()  # must be before bc for 3d structures with monitor lines
            self._map_bc()
            self._map_obstructions()
        except:  # pragma: no cover  # noqa
            self._logger.exception('Error generating snap.')  # pragma: no cover

    def map_monitor_for_advanced_sim_dlg(self):
        """Maps just the monitor data for the advanced simulation dialog."""
        try:
            self._map_monitor()
        except:  # pragma: no cover  # noqa
            self._logger.exception('Error generating snap.')  # pragma: no cover

    def _map_materials(self):
        """Maps the materials from the material coverage to the mesh."""
        self._map_material_coverage()
        self._map_raster_materials()
        comp_to_cells = self.material_comp_id_to_grid_cell_ids
        if comp_to_cells and 0 in comp_to_cells and len(comp_to_cells[0]) > 0:
            cells = [x + 1 for x in self.material_comp_id_to_grid_cell_ids[0]]
            self._logger.warning(
                f'The following elements were assigned to the "unassigned" material.\n'
                f'Element ids: {cells}.'
            )

    def _map_material_coverage(self):
        """Maps the materials from the material coverage to the mesh."""
        if self.material_coverage is None or not self.material_component_file:
            return
        self._logger.info('Mapping materials coverage to mesh.')
        mapper = MaterialMapper(self, wkt=self.grid_wkt, sediment=False, generate_snap=self._generate_snap)
        mapper.mapped_comp_uuid = self.mapped_material_uuid
        mapper.mapped_material_display_uuid = self.mapped_material_display_uuid
        self.material_comp_id_to_grid_cell_ids = mapper._poly_to_cells
        mat_data = MaterialData(self.material_component_file)
        self.material_names = mat_data.materials.to_dataframe()['Name'].tolist()
        self.material_comp_ids = mat_data.materials.to_dataframe()['id'].tolist()
        do_comp, comp = mapper.do_map()
        self.material_mannings = mapper._mannings_n
        if do_comp is not None:
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()], 'mapped_material_component'))
        self._logger.info('Finished mapping materials coverage to mesh.')

    def _map_nlcd_to_ugrid(self):
        """Get the cells to map the NLCD raster to."""
        ug = self._query_helper.co_grid.ugrid
        # get a set of all cells already assigned based on material polygons
        if self.material_comp_id_to_grid_cell_ids and 0 in self.material_comp_id_to_grid_cell_ids:
            idxs = self.material_comp_id_to_grid_cell_ids[0]
            self.material_comp_id_to_grid_cell_ids[0] = []
        else:
            idxs = list(range(ug.cell_count))

        raster_file = self._query_helper.nlcd_raster_file
        locs = [ug.get_cell_centroid(i)[1] for i in idxs]
        locs_wkt = self._query_helper.grid_wkt
        locs_vert = ''  # don't need vertical coordinate
        vals, _ = ru.interpolate_raster_to_points(
            raster_file=raster_file,
            xy_locations=locs,
            xy_locations_wkt=locs_wkt,
            xy_locations_vertical_units=locs_vert,
            resample_alg='nearest_neighbor'
        )

        nlcd_cell_classes = {}
        unassigned_cells = []
        nlcd_to_manning = _nlcd_to_manning()
        for idx, val in zip(idxs, vals):
            val = int(val)
            if val not in nlcd_to_manning:
                unassigned_cells.append(idx)
                continue
            cell_list = nlcd_cell_classes.get(val, [])
            cell_list.append(idx)
            nlcd_cell_classes[val] = cell_list

        return nlcd_cell_classes, unassigned_cells

    def _update_materials_from_nlcd(self, nlcd_cell_classes, unassigned_cells):
        """Update the materials based on the NLCD raster."""
        if not self.material_names:
            self.material_names = ['unassigned']
            self.material_comp_ids = [0]
            self.material_mannings = [0.02]
            self.material_comp_id_to_grid_cell_ids = {}

        next_comp_id = max(self.material_comp_ids) + 1
        if unassigned_cells:
            self.material_names.append('NLCD_background')
            self.material_comp_ids.append(next_comp_id)
            self.material_comp_id_to_grid_cell_ids[next_comp_id] = unassigned_cells
            self.material_mannings.append(self._query_helper.nlcd_background_manning_n)
            self._logger.warning(
                f'The following elements were assigned to the "NLCD_background" material.\n'
                f'Element ids: {unassigned_cells}.'
            )
            next_comp_id += 1

        nlcd_to_manning = _nlcd_to_manning()
        keys = list(nlcd_cell_classes.keys())
        keys.sort()
        for key in keys:
            nlcd_category = key
            cell_list = nlcd_cell_classes[key]
            self.material_names.append(f'NLCD_{nlcd_category}')
            self.material_comp_ids.append(next_comp_id)
            self.material_comp_id_to_grid_cell_ids[next_comp_id] = cell_list
            next_comp_id += 1
            self.material_mannings.append(nlcd_to_manning[nlcd_category])

    def _map_raster_materials(self):
        """Maps the raster materials from the material coverage to the mesh."""
        if not self._query_helper.nlcd_raster_file:
            return
        self._logger.info('Mapping NLCD raster to mesh.')
        nlcd_cell_classes, unassigned_cells = self._map_nlcd_to_ugrid()
        self._update_materials_from_nlcd(nlcd_cell_classes, unassigned_cells)
        self._logger.info('Finished mapping NLCD raster to mesh.')

    def _map_sed_materials(self):
        """Maps the materials from the material coverage to the mesh."""
        if self.sed_material_coverage is None or not self.sed_material_component_file:
            return
        self._logger.info('Mapping sediment materials coverage to mesh.')
        mapper = MaterialMapper(self, wkt=self.grid_wkt, sediment=True, generate_snap=self._generate_snap)
        mapper.mapped_comp_uuid = self.mapped_sed_material_uuid
        mapper.mapped_material_display_uuid = self.mapped_sed_material_display_uuid
        self.sed_material_comp_id_to_grid_cell_ids = mapper._poly_to_cells
        mat_data = MaterialData(self.sed_material_component_file)
        self.sed_material_names = mat_data.materials.to_dataframe()['Name'].tolist()
        self.sed_material_comp_ids = mat_data.materials.to_dataframe()['id'].tolist()
        do_comp, comp = mapper.do_map()
        self.sed_material_data = mapper.sediment_lines
        if do_comp is not None:
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()], 'mapped_sed_material_component'))
        self._logger.info('Finished mapping sediment materials coverage to mesh.')

    def _map_bc(self):
        """Maps the bcs from the bc coverage to the mesh."""
        if self.bc_coverage is None or not self.bc_component_file:
            return
        self._logger.info('Mapping bc coverage to mesh.')
        mapper = BcMapper(self, wkt=self.grid_wkt, generate_snap=self._generate_snap)
        mapper.ceiling_file = self.ceiling_file
        mapper.bc_mapped_comp_uuid = self.bc_mapped_comp_uuid
        mapper.bc_mapped_comp_display_uuid = self.bc_mapped_comp_display_uuid
        mapper.structures_3d_weirs = self.structures_3d_weirs
        self.bc_arc_id_to_grid_ids = mapper.arc_id_to_grid_ids
        self.bc_arc_id_to_grid_pts = mapper.arc_id_to_grid_pts
        self.bc_arc_id_to_node_string_lengths = mapper.arc_id_to_node_string_length
        self.bc_arc_id_to_comp_id = mapper._arc_id_to_comp_id
        self.bc_arc_id_to_bc_param = mapper._arc_id_to_bc_param
        self.bc_arc_id_to_bc_id = mapper._arc_id_to_bc_id
        self.bc_id_to_structures = mapper._structures
        self.bc_hy8_file = mapper._bc_component.hy8_file
        self.bc_3d_structures = mapper.out_3d_structures
        do_comp, comp = mapper.do_map()
        if do_comp is not None:
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()], 'mapped_bc_component'))
        self._logger.info('Finished mapping bc coverage to mesh.')

    def _map_monitor(self):
        """Maps the arcs from the monitor coverage to the mesh."""
        if self.monitor_coverage:
            self._logger.info('Mapping monitor coverage to mesh.')
        comp_path = self.component_folder
        mapper = MonitorMapper(
            self,
            wkt=self.grid_wkt,
            generate_snap=self._generate_snap,
            comp_path=comp_path,
            structures_3d_monitor=self.structures_3d_monitor
        )
        mapper.mapped_comp_uuid = self.monitor_mapped_comp_uuid
        mapper.mapped_comp_display_uuid = self.monitor_mapped_comp_display_uuid
        self.monitor_points = mapper._monitor_points
        self.monitor_arc_ids_to_grid_cell_ids = mapper.arc_id_to_grid_ids
        do_comp, comp = mapper.do_map()
        if do_comp is not None:
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()], 'mapped_monitor_component'))
        if self.monitor_coverage:
            self._logger.info('Finished mapping monitor coverage to mesh.')
        self.structures_3d_monitor = mapper.out_3d_structures

    def _map_obstructions(self):
        """Maps the arcs from the obstructions coverage to the mesh."""
        if self._generate_snap:
            return
        if self.obstructions_coverage is None:
            return
        self._logger.info('Mapping obstructions coverage to mesh.')
        mapper = ObstructionsMapper(self)
        mapper.do_map()
        self.obstructions_decks = mapper._obstructions_decks
        self.obstructions_piers = mapper._obstructions_piers
        self._logger.info('Finished mapping obstructions coverage to mesh.')


def _nlcd_to_manning():
    """Get a dict of NLCD values to Manning's n values."""
    return {
        11: 0.02,  # Open Water
        12: 0.01,  # Perennial Ice/Snow
        21: 0.02,  # Developed, Open Space
        22: 0.05,  # Developed, Low Intensity
        23: 0.1,  # Developed, Medium Intensity
        24: 0.15,  # Developed, High Intensity`
        31: 0.09,  # Barren Land (Rock/Sand/Clay)
        41: 0.1,  # Deciduous Forest
        42: 0.11,  # Evergreen Forest
        43: 0.1,  # Mixed Forest
        51: 0.04,  # Dwarf Scrub
        52: 0.05,  # Shrub/Scrub
        71: 0.034,  # Grassland/Herbaceous
        72: 0.03,  # Sedge/Herbaceous
        73: 0.027,  # Lichens
        74: 0.025,  # Moss
        81: 0.033,  # Pasture/Hay
        82: 0.037,  # Cultivated Crops
        90: 0.1,  # Woody Wetlands
        91: 0.1,  # Palustrine Forested Wetland
        92: 0.048,  # Palustrine Scrub/Shrub Wetland
        93: 0.1,  # Estuarine Forested Wetland
        94: 0.048,  # Estuarine Scrub/Shrub Wetland
        95: 0.045,  # Emergent Herbaceous Wetlands
        96: 0.045,  # Palustrine Emergent Wetland
        97: 0.045,  # Estuarine Emergent Wetland
        98: 0.015,  # Palustrine Aquatic Bed
        99: 0.015  # Estuarine Aquatic Bed
    }
