"""Classes to represent SRH model run progress output files."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import shlex

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as io_util

# 4. Local modules

# Columns of interest in the tabular output files.
RES_COLUMN_TIME = 0  # Residual file time column
RES_COLUMN_H = 2  # Residual file H column

INF_COLUMN_TIME_IDX = 0  # INF progress file columns
INF_COLUMN_TIME = 1

PT_COLUMN_TIME = 0  # Monitor point file time column
PT_COLUMN_GROUND = 3  # Monitor point file bed elevation column
PT_COLUMN_WSE = 4  # Monitor point file water surface elevation column

ARC_COLUMN_TIME = 0  # Monitor arc file time column
ARC_COLUMN_Q = 1  # Monitor arc file flow column
ARC_COLUMN_QS = 2  # Monitor arc file sediment flux table


class PointFile:
    """Class to hold plot data for a monitor point."""
    def __init__(self, file_pos):
        """Constructor.

        Args:
            file_pos (:obj:`int`): Current position in the arc's output file
        """
        self.file_pos = file_pos
        self.db_table = {}


class ArcFile:
    """Class to hold plot data for a monitor arc."""
    def __init__(self, file_pos, db_table):
        """Constructor.

        Args:
            file_pos (:obj:`int`): Current position in the arc's output file
            db_table (:obj:`str`): The plot database table name for this arc
        """
        self.file_pos = file_pos
        self.db_table = db_table


class InfFile:
    """Class to hold plot data for a mass balance output file."""
    def __init__(self):
        """Constructor."""
        self.start = 0.0
        self.end = 1.0
        self.run_duration = 1.0
        self.inf_file = ""
        self.inf_file_pos = 0
        self.progress_percent = 0
        self.num_inserted_to_db = 0
        self.current_time = 0.0
        self.using_restart = False
        self._first_read_INF = True
        self.data = []

    def update_progress(self):
        """Compute the model run progress percent."""
        try:
            with open(self.inf_file, 'r') as f:
                f.seek(self.inf_file_pos)
                for line in f:
                    vals = shlex.split(line)
                    if len(vals) > 5:
                        try:
                            # time_idx = int(vals[INF_COLUMN_TIME_IDX])
                            time_val = float(vals[INF_COLUMN_TIME])
                            if self.using_restart and self._first_read_INF:
                                self._first_read_INF = False
                                self.start = time_val
                                self.run_duration = self.end - self.start
                            net_q = float(vals[INF_COLUMN_TIME + 1])
                            mass_error = float(vals[INF_COLUMN_TIME + 2])
                            cumu_mass_error = float(vals[INF_COLUMN_TIME + 3])
                            wet_elements = int(vals[INF_COLUMN_TIME + 4])
                            self.data.append((time_val, net_q, mass_error, cumu_mass_error, wet_elements))
                            self.current_time = time_val - self.start
                            self.progress_percent = int((self.current_time / float(self.run_duration)) * 100.0)
                        except ValueError:
                            pass
                        self.inf_file_pos += len(line)
        except FileNotFoundError:
            pass  # File might not exist yet

    def reset(self):
        """Clear all run data, set progress back to 0.0, and reset file position markers."""
        io_util.removefile(self.inf_file)
        self.inf_file_pos = 0
        self.progress_percent = 0
        self.num_inserted_to_db = 0
        self.data = []
