"""Script used to run SRH."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import glob
import json
import os
from pathlib import Path
import shutil
from typing import Optional

# 2. Third party modules
import h5py
import orjson
import pandas as pd

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand, XmsEnvironment as XmEnv
from xms.api.tree import tree_util
from xms.components.bases.run_base import RunBase
from xms.core.filesystem import filesystem
from xms.data_objects.parameters import Dataset, FilterLocation
from xms.guipy.data.plot_and_table_data_base import np_arrays_from_file
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.srh.components.parameters.parameters_manager import ParametersManager
from xms.srh.components.sim_query_helper import SimQueryHelper
from xms.srh.model.srh_post_process import update_xmdfc_file


def get_sms_srh_exes():
    """Get the SMS SRH exes from the registery.

    Returns:
        (dict): SRH exes
    """
    exes = {}
    settings = SettingsManager(python_path=False)
    package = 'File_Location_Preferences'
    key = 'SRH-2D - PreSRH-2D'
    val = settings.get_setting(package, key, '')
    exes[key] = val
    key = 'SRH-2D - SRH-2D'
    val = settings.get_setting(package, key, '')
    exes[key] = val
    key = 'SRH-2D - HY-8'
    val = settings.get_setting(package, key, '')
    exes[key] = val
    return exes


class RunSRH(RunBase):
    """Says what SRH executables will run."""
    def __init__(self, dummy_mainfile=''):
        """Constructor."""
        super().__init__()
        self.proj_name = ''
        self.case_name = ''
        self._scenarios = []
        self.query_helper = None
        self.build_vertex = None
        self._sim_uuid: Optional[str] = None
        self._model_order = 0
        self._run_id = -1
        self._sub_dir = ''
        self._hydro_file = ''
        self._using_pest = False
        self._sim_run_dir = ''
        self._file_location = ''
        self._main_file = ''
        self._exes = []
        self._query = None
        self._model_run_dict = {}

    @property
    def sim_uuid(self) -> str:
        """The UUID of the simulation, not its component."""
        if not self._sim_uuid and self._query:
            self._sim_uuid = self._query.current_item_uuid()
        return self._sim_uuid

    def set_sim_uuid(self, sim_uuid: str):
        """Setter for the sim_uuid."""
        if sim_uuid:
            self._sim_uuid = sim_uuid

    def _check_srh_version_in_file_preferences(self, run):
        """Fix the default version of SRH that SMS references.

        This is to address upgrade path from SMS 13.1.2 to later versions that use SRH 3.3.

        Args:
            run (:obj:`int`): The index of the current run.

        Returns:
            None or (:obj:`list`) of ExecutableCommand objects for a run.
        """
        err = False
        exes = get_sms_srh_exes()
        old_exes = [
            'srh2d_3.2_console.exe', 'srh-2d_console_v340.exe', 'srh-2d_console_v350.exe', 'srh-2d_console_v360.exe',
            'srh-2d_console_v361.exe'
        ]
        for o in old_exes:
            if exes['SRH-2D - SRH-2D'].lower().endswith(o):
                err = True
        if exes['SRH-2D - PreSRH-2D'].lower().endswith('srh2d_pre_3.2.exe'):
            err = True
        rval = None
        if err:
            cmd = os.path.join(os.path.dirname(__file__), 'srh_version_check.py')
            srh_version_check = ExecutableCommand(
                executable=cmd,
                model='SRH-2D',
                executable_order=1,
                executable_is_script=True,
                display_name='SRH-2D Version Check FAILED',
                progress_script='srh_version_check_progress.py',
                run_weight=100
            )
            srh_version_check.set_run_group_and_id(run, 0)
            rval = [srh_version_check]
        return rval, exes

    def _setup_query_helper(self, query):
        """Sets up the SimQueryHelper.

        Args:
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with SMS.
        """
        self._query = query
        if not self.query_helper:
            self.query_helper = SimQueryHelper(query, at_sim=True, sim_uuid=self.sim_uuid)
            self.query_helper.get_sim_data()

    def get_executables(self, sim, query, file_location):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with SMS.
            file_location (:obj:`str`): The location of input files for the simulation.

        Returns:
            (:obj:`list[xms.api.dmi.ExecutableCommand]`):
                The executable objects to run and the action requests that go with it.
        """
        self._sim_run_dir = file_location
        self._setup_query_helper(query)
        sim_comp = None
        if not self.proj_name:
            sim_comp = self._get_proj_name_case_name(query)

        executables = []
        param_data = ParametersManager.read_parameter_file(sim_comp.main_file)
        if param_data and param_data['use_parameters']:
            if param_data['run_type'] == 'Scenarios':
                # Add executables for each run
                run_count = param_data['run_count']
                for run in range(run_count):
                    run_name = param_data['runs']['Run'][run]
                    self.case_name = run_name
                    run_dir = os.path.join(file_location, run_name)
                    executables.extend(
                        self._get_executable_commands(sim, query, run_dir, using_params=True, run_id=run)
                    )
            else:
                executables.extend(
                    self._get_executable_commands(
                        sim, query, file_location, using_pest=True, main_file=sim_comp.main_file
                    )
                )
        else:
            executables.extend(self._get_executable_commands(sim, query, file_location))
        return executables

    def _get_executable_commands(
        self, sim, query, file_location, using_params=False, using_pest=False, run_id=0, main_file=''
    ):
        """Returns list of ExecutableCommand objects for a run.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with SMS.
            file_location (:obj:`str`): The location of input files for the simulation.
            using_params (:obj:`bool`): True if using parameters.
            using_pest (:obj:`bool`): True if using pest.
            run_id (:obj:`int`): The index of the current run.
            main_file (:obj:`str`): The location of the main_file for the simulation component

        Returns:
            See description.

        """
        self._run_id = run_id
        rval, self._exes = self._check_srh_version_in_file_preferences(self._run_id)
        if rval is not None:
            return rval
        self._file_location = file_location
        self._clear_output_folder(file_location)
        self._hydro_file = os.path.join(file_location, self.proj_name + '.srhhydro')
        self._model_order = 1
        self._sub_dir = '' if not using_params else f'./{self.case_name}/'
        self._using_pest = using_pest
        self._main_file = main_file
        # If the hydro file does not exist, give up and tell the user to quit being dumb.
        if not os.path.isfile(self._hydro_file):
            XmEnv.report_error(
                f'Unable to run SRH-2D because "{self._hydro_file}" was not found. Ensure the simulation has been '
                'exported.'
            )
            return []

        sim_item = tree_util.find_tree_node_by_uuid(query.project_tree, self.sim_uuid)
        self._model_run_dict = {}
        self._model_run_dict['sim_name'] = sim_item.name
        self._model_run_dict['case_name'] = self.case_name
        pest, pest_post = self._pest_model()
        srh_run = self._new_srh_run()

        # Set up Monitor point and line plots
        cov_item = tree_util.descendants_of_type(
            sim_item,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            only_first=True,
            recurse=False,
            coverage_type='Monitor',
            model_name='SRH-2D'
        )
        sed_on = self.query_helper.sim_component.data.enable_sediment
        if cov_item and run_id == 0:
            monitor_cov = query.item_with_uuid(cov_item.uuid)
            monitor_pts = monitor_cov.get_points(FilterLocation.PT_LOC_DISJOINT)

            # Monitor Point Plot
            for pt in monitor_pts:
                self.add_plot_data("monitorPtWSEPlot", f"Pt{pt.id} WSE", on=not sed_on)
                self.add_plot_data("monitorPtZPlot", f"Pt{pt.id} Z", on=sed_on)

            # Monitor Line Q Plot
            self.color_idx = 0
            for arc in monitor_cov.arcs:
                self.add_plot_data("monitorLineQPlot", f"Arc{arc.id}", on=True)

            if sed_on:
                # Monitor Line QS Plot
                self.color_idx = 0
                for arc in monitor_cov.arcs:
                    self.add_plot_data("monitorLineQsPlot", f"Arc{arc.id}", on=True)

        runs = []
        if pest:
            runs.extend([pest, pest_post])
        elif srh_run:
            runs.append(srh_run)
        # read solution
        load_sol = self.get_solution_load_actions(sim, query, file_location, all_scenarios=False)
        runs[-1].add_solution_file(load_sol[0])
        return runs

    def _new_srh_run(self):
        """Return the SRH run model script.

        Returns:
            (:obj:`ExecutableCommand`): see above
        """
        self._hy8_model()
        self._srh_pre_model()
        self._srh_model()
        self._srh_post_model()
        cmd = os.path.join(os.path.dirname(__file__), 'srh_scenario_run.py')
        main = ExecutableCommand(
            executable=cmd,
            model='SRH-2D',
            executable_order=self._model_order,
            display_name=f'SRH-2D - {self.case_name}',
            run_weight=100,
            progress_script='srh_progress.py',
            plot_group='progressGroup',
            executable_is_script=True
        )
        main.set_run_group_and_id(self._run_id, 0)
        main.add_progress_arg(str(self._run_id + 1))
        self._model_order += 1

        json_file = os.path.join(self._file_location, f'srh_run_{self._model_order}.json')
        with open(json_file, 'w') as f:
            json.dump(self._model_run_dict, f)
        main.add_commandline_arg(f'-j {json_file}')
        return main

    def _srh_post_model(self):
        """Add SRH Post model to model_run_dict."""
        if self.query_helper.using_ugrid:
            return

        self._model_run_dict['srh_post_input'] = f'{os.path.join(self._file_location, "srh_post.json")}'

    def _srh_model(self):
        """Add the SRH model to the model_run_dict."""
        input_file = f'{self.case_name}.dat'
        if self._sub_dir:
            input_file = f'{self._sub_dir}{self.case_name}.dat'
        self._model_run_dict['srh_exe'] = os.path.normpath(self._exes["SRH-2D - SRH-2D"])
        self._model_run_dict['srh_exe_input'] = input_file

    def _srh_pre_model(self):
        """Add the SRH Pre model to the model_run_dict."""
        if self._sub_dir:
            self._model_run_dict['run_in_sub_dir'] = True
            input_file = f'{self._sub_dir}{self.proj_name}.srhhydro'
            srh_pre_input = input_file
        else:
            if self.query_helper.sim_component.data.advanced.run_in_partial_mode:
                srh_pre_input = f'{self.case_name}'
                sif_file = os.path.join(self._file_location, f'{self.case_name}_SIF.dat')
                if not os.path.isfile(sif_file):
                    sof_file = os.path.join(self._file_location, f'{self.case_name}_SOF.dat')
                    if os.path.isfile(sof_file):
                        shutil.copyfile(sof_file, sif_file)
            else:
                srh_pre_input = f'{self.proj_name}.srhhydro'
        self._model_run_dict['srh_pre_exe'] = os.path.normpath(self._exes["SRH-2D - PreSRH-2D"])
        self._model_run_dict['srh_pre_input'] = srh_pre_input

    def _pest_model(self):
        """Return the PEST model if we are running PEST.

        Returns:
            (:obj:`tuple(ExecutableCommand)`): see above
        """
        if not self._using_pest:
            return None, None
        # Add PEST command
        pest = ExecutableCommand(
            executable='PEST',
            model='SRH-2D',
            executable_order=self._model_order,
            display_name='PEST SRH-2D',
            run_weight=100,
            progress_script='srh_pest_progress.py',
            plot_group='pest_plot_group'
        )
        pest.set_run_group_and_id(self._run_id, 0)
        self._model_order += 1

        with open(os.path.join(self._file_location, 'srh_pest.json'), 'rb') as f:
            json_dict = orjson.loads(f.read())
        for parameter in json_dict['PARAMETER_NAMES']:
            self.add_plot_data('pest_parameters', f'{parameter}', on=False)

        pest.add_commandline_arg(f'{self.proj_name}.pst')

        # Save final run with optimized parameters
        cmd = os.path.join(os.path.dirname(__file__), 'srh_pest_post.py')
        pest_post = ExecutableCommand(
            executable=cmd,
            model='SRH-2D',
            executable_order=self._model_order,
            display_name='PEST SRH-2D Final Run Setup',
            run_weight=10,
            executable_is_script=True
        )
        pest_post.set_run_group_and_id(self._run_id, 0)
        pest_post.add_commandline_arg(f'-f {self._hydro_file}')
        pest_post.add_commandline_arg(f'-m {self._main_file}')
        self._model_order += 1
        return pest, pest_post

    def _hy8_model(self):
        """Determine if HY-8 must run before SRH."""
        with open(self._hydro_file, 'r') as f:
            lines = f.readlines()
        for line in lines:
            if line.lower().startswith('hy8file'):
                val = line.split()[1].strip('"')
                hy8_file = os.path.normpath(os.path.join(os.path.dirname(self._hydro_file), val))
                if not os.path.isfile(hy8_file):
                    msg = f'Unable to find HY-8 file:\n "{hy8_file}"\nHY-8 will not run with this simulation.'
                    XmEnv.report_error(msg)
                    return

                hy8_file = filesystem.compute_relative_path(self._sim_run_dir, hy8_file)
                self._model_run_dict['hy8_input'] = hy8_file
                self._model_run_dict['hy8_exe'] = self._exes['SRH-2D - HY-8']

    def get_solution_load_actions(self, sim, query, filelocation, all_scenarios=True):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with GMS.
            filelocation (:obj:`str`): The location of input files for the simulation.
            all_scenarios (:obj:`bool`): indicates if we should get the files for all scenarios

        Returns:
            (:obj:`list[xms.api.dmi.ExecutableCommand]`):
                The executable objects to run and the action requests that go with it.
        """
        self._setup_query_helper(query)
        if not self.proj_name:
            self._get_proj_name_case_name(query)

        rval = []
        if not all_scenarios or len(self._scenarios) < 1:
            all_scenarios = False
            self._scenarios = ['']
        ug_sol = ''
        if self.query_helper.using_ugrid:
            ug_sol = 'C'
        for case in self._scenarios:
            if not all_scenarios:
                file_name = os.path.join(filelocation, f'{self.case_name}_XMDF{ug_sol}.h5')
            else:
                file_name = os.path.join(filelocation, case, f'{case}_XMDF{ug_sol}.h5')
            load_sol = ActionRequest(
                modality='MODAL',
                class_name='RunSRH',
                module_name='xms.srh.srh_run',
                method_name='read_solution',
                parameters={'solution_file': file_name}
            )
            rval.append(load_sol)
        return rval

    def _get_proj_name_case_name(self, query):
        """Does several things.

        Args:
            (:obj:`query`):

        Returns:
            sim_comp (:obj:`SimComponent`): The SimComponent.
        """
        self.proj_name = os.path.splitext(os.path.basename(query.xms_project_path))[0]
        sim_comp = self.query_helper.sim_component

        # Set case name
        self.case_name = sim_comp.data.hydro.case_name
        # set scenario names if they exist
        self._scenarios = []
        par_data = ParametersManager.read_parameter_file(sim_comp.main_file)
        if par_data is not None:
            if par_data['use_parameters'] and par_data['run_type'] != 'Calibration':
                self._scenarios = par_data['runs']['Run']

        return sim_comp

    def _clear_output_folder(self, filelocation):
        """Clears the output folder in anticipation of a model run.

        Args:
            filelocation (:obj:`str`): The location of input files for the simulation.

        """
        sim = None if self.query_helper is None else self.query_helper.sim_component
        skip_files = ['_SIF.dat', '_SOF.dat', '_RST1.dat']
        if os.path.isfile('c:/temp/retain_dip_file.txt'):
            skip_files.append('_DIP.dat')
        elif sim is not None and sim.data.hydro.initial_condition == 'Restart File':
            skip_files.append('_DIP.dat')
        if sim is not None and sim.data.hydro.initial_condition == 'Restart File':
            skip_files.extend(['_RES.dat', '_XMDFC.h5'])
            xmdfc = os.path.join(filelocation, f'{self.case_name}_XMDFC.h5')
            h5_file = h5py.File(xmdfc, 'w')
            file_type = 'Xmdf'
            ascii_list = [file_type.encode("ascii", "ignore")]
            h5_file.create_dataset('File Type', shape=(1, ), dtype='S5', data=ascii_list)
            h5_file.create_dataset('File Version', data=99.99, dtype='f')
            json_file = os.path.join(filelocation, 'srh_post.json')
            with open(json_file, 'rb') as f:
                json_dict = orjson.loads(f.read())
                json_dict['RESTART'] = True
            with open(json_file, 'wb') as f:
                f.write(orjson.dumps(json_dict))

        for _, _, files in os.walk(filelocation):
            for f in files:
                if f.endswith(tuple(skip_files)):
                    continue
                _, ext = os.path.splitext(f)
                if ext.upper() in ['.DAT', '.H5']:
                    filesystem.removefile(os.path.join(filelocation, f))

        out_misc_dirs = [str(f.absolute()) for f in Path(filelocation).rglob('Output_MISC')]
        for out_misc in out_misc_dirs:
            if os.path.isdir(out_misc):
                shutil.rmtree(out_misc, ignore_errors=True)

    def read_solution(self, query, params, win_cont):
        """Reads the SRH Solution.

        Args:
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with GMS.
            params (:obj:`dict`): Generic map of parameters. Contains the structures for various components that
             are required for adding vertices to the Query Context with Add().
            win_cont (:obj:`QWidget`): Parent window
        """
        ds_file_name = params[0]['solution_file']
        json_file = os.path.join(os.path.dirname(ds_file_name), 'srh_post.json')
        _update_3d_structure_output_file(json_file)
        with open(json_file, 'rb') as f:
            json_dict = orjson.loads(f.read())
        if 'GRID_UUID' in json_dict:
            update_xmdfc_file(ds_file_name, json_dict['GRID_UUID'])

        if not os.path.isfile(ds_file_name):
            ds_dir = os.path.dirname(ds_file_name)
            base = os.path.splitext(os.path.basename(ds_file_name))[0]
            base += 'C.h5'
            cell_ds_file_name = os.path.join(ds_dir, base)
            if not os.path.isfile(cell_ds_file_name):
                warn = [
                    ('Warning', f'Solution file does not exist: {cell_ds_file_name}.'),
                    ('Warning', 'Run SRH-2D to generate this file.')
                ]
                return warn, []
            else:
                srh_post_json = os.path.join(ds_dir, 'srh_post.json')
                if not os.path.isfile(srh_post_json):
                    warn = [
                        ('Warning', f'PostSRH-2D missing input file: {srh_post_json}.'),
                        ('Warning', 'Export the simulation from SMS to create the input file.')
                    ]
                    return warn, []
                else:
                    from xms.srh.model.srh_post_runner import run_post_process
                    cur_dir = os.getcwd()
                    os.chdir(ds_dir)
                    run_post_process(win_cont=win_cont)
                    os.chdir(cur_dir)

        rst_files = glob.glob(f'{os.path.dirname(ds_file_name)}{os.path.sep}*_RST*.dat')
        if 'RESTART' in json_dict:
            rst_files = [f for f in rst_files if not f.endswith('_RST1.dat')]
        if rst_files:
            rst_times = [os.path.getmtime(rst) for rst in rst_files]
            max_time = max(rst_times)
            for rst_time, rst_file in zip(rst_times, rst_files):
                if rst_time < max_time:
                    os.remove(rst_file)

        scenario = ''
        if os.path.basename(os.path.dirname(os.path.dirname(ds_file_name))) != 'SRH-2D':
            scenario = os.path.basename(os.path.dirname(ds_file_name))

        data_loc = 'NODE'
        if ds_file_name.endswith('XMDFC.h5'):
            data_loc = 'CELL'

        using_folders = False
        dataset_kwargs = []
        solution_file = h5py.File(ds_file_name, 'r')
        dataset_groups = solution_file.keys()
        if 'Datasets' in solution_file:
            dataset_groups = solution_file['Datasets'].keys()
        for ds in dataset_groups:
            if ds.lower() in ['guid', 'file version', 'file type']:
                continue
            ds_name = ds
            if 'Datasets' in solution_file:
                ds_name = f'Datasets/{ds}'
            dataset = Dataset(ds_file_name, ds_name, data_loc)

            dset_kwargs = {'do_dataset': dataset}
            if scenario:
                dset_kwargs['folder_path'] = scenario
                using_folders = True
            if ds.startswith('Max_') or ds.startswith('time_of_Max_'):
                path = ''
                if 'folder_path' in dset_kwargs:
                    path = dset_kwargs['folder_path']
                    path = path + '/'
                path = path + 'Max'
                if ds.startswith('time_of_Max_'):
                    path = path + '/time_of_max'
                dset_kwargs['folder_path'] = path
                using_folders = True
            dataset_kwargs.append(dset_kwargs)

        # This is just so the datasets show up in a consistent order in the tree.
        if using_folders:
            dataset_kwargs.reverse()

        for dset in dataset_kwargs:
            query.add_dataset(**dset)
        return [], []


def _update_3d_structure_output_file(json_file_path):
    """Update the 3D structure output file.

    Args:
        json_file_path (str): The path to the json file.
    """
    model_dir = os.path.dirname(json_file_path)
    struct_index_file = os.path.join(model_dir, 'structure_index.txt')
    if not os.path.isfile(struct_index_file):
        return
    structs_dict = _dict_from_struct_index(struct_index_file)
    if len(structs_dict) < 1:
        return
    base_path = os.path.join(model_dir, 'Output_MISC')
    # get all files associated with 3d structures
    for _, v in structs_dict.items():
        struct_index = v.get('structure_index', -1)
        monitor_index = v.get('monitor_index', -1)
        struct_pattern = f'*_INTERNAL{struct_index}.dat'
        struct_file = [str(f.absolute()) for f in Path(base_path).rglob(struct_pattern)]
        mon_pattern = f'*_LN{monitor_index}.dat'
        monitor_file = [str(f.absolute()) for f in Path(base_path).rglob(mon_pattern)]
        if len(struct_file) == 1 and len(monitor_file) == 1:
            s_headings, struct_data = np_arrays_from_file(struct_file[0])
            m_headings, mon_data = np_arrays_from_file(monitor_file[0])
            if len(struct_data) < 1 or len(mon_data) < 1:
                continue
            if 'Q-Structure' in s_headings[2]:
                continue  # already updated
            shutil.copyfile(struct_file[0], struct_file[0] + '.bak')
            s_headings[1] = s_headings[1].replace('Discharge', 'Q-Overtopping')
            q_struct = s_headings[1].replace('Q-Overtopping', 'Q-Structure')
            q_total = s_headings[1].replace('Q-Overtopping', 'Q-Total')
            s_headings = s_headings[:2] + [q_struct, q_total] + s_headings[2:]
            mon_data[1] = [abs(mon_data[1][i]) for i in range(len(mon_data[1]))]
            total_flow = [struct_data[1][i] + mon_data[1][i] for i in range(len(struct_data[1]))]
            struct_data = struct_data[:2] + [mon_data[1], total_flow] + struct_data[2:]
            with open(struct_file[0], 'w') as f:
                f.write('//\n')
                f.write('// Information at the Internal Boundary: WEIR Option\n')
                f.write('//\n')
                f.write(' '.join(s_headings))
                f.write('\n')
                for row in range(len(struct_data[0])):
                    f.write(' '.join([f'{struct_data[i][row]:.8E}' for i in range(len(struct_data))]))
                    f.write('\n')


def _dict_from_struct_index(struct_index_file):
    """Put information about 3d structures into a dictionary from the 'structure_index.txt' file.

    Args:
        struct_index_file (str): The path to the structure index file.
    """
    df = pd.read_csv(struct_index_file)
    structs_dict = {}
    for _, row in df.iterrows():
        arc_id = abs(int(row['arc_id']))
        cov_uuid = row['coverage_uuid']
        key = f'{arc_id}_{cov_uuid}'
        item_dict = structs_dict.get(key, {})
        if row['structure_type'] == 'INTERNAL':
            item_dict['structure_index'] = row['structure_index']
        elif row['structure_type'] == 'MONITOR_MIDDLE':
            item_dict['monitor_index'] = row['structure_index']
        else:
            continue
        structs_dict[key] = item_dict
    return structs_dict
