"""SwmmInletFlowsFromSrhMonitorLines class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import numpy as np
import pandas as pd

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.data_objects.parameters import FilterLocation
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.plot_and_table_dialog import np_arrays_from_file
from xms.srh.components.monitor_component import MonitorComponent
from xms.srh.components.sim_component import SimComponent as SrhSimComponent
from xms.swmm.components.coverage_component import get_node_groups
from xms.swmm.data.model import get_swmm_model

# 4. Local modules
from xms.srh_swmm_coupler.data.model import get_node_coverage, get_srh_coverage
from xms.srh_swmm_coupler.dmi.xms_data import CouplerData
from xms.srh_swmm_coupler.gui.difference_viewer import DifferenceViewer
from xms.srh_swmm_coupler.tools.srh_internal_sink_flows_from_swmm_inlet_pipe_flows import make_comparison


class SwmmInletFlowsFromSrhMonitorLines:
    """SwmmInletFlowsFromSrhMonitorLines class."""

    def __init__(self, query: Query, table: list, damping_factor: float):
        """Initializes the class.

        Args:
            query (Query): Interprocess communication object.
            table (list): Table linking each inlet to a sink BC arc and a monitor line arc from SRH.
            damping_factor (float): A multiplication factor for assigning SRH monitor line flows to SWMM inlet flows.
        """
        self._coupler_data = CouplerData(query)
        self._query = query
        self._node_cov, self._node_component = get_node_coverage(self._coupler_data)
        self._monitor_cov, self._monitor_component = get_srh_coverage(self._coupler_data, 'SRH-2D', 'Monitor',
                                                                      MonitorComponent, 'Monitor_Component')
        self._table = table
        self._damping_factor = damping_factor

    def run(self):
        """Runs the tool."""
        arcs = self._monitor_cov.arcs
        # get the path to the SRH output files
        srh_sim_tree_item = self._coupler_data.srh_sim[0]
        swmm_sim_tree_item = self._coupler_data.swmm_sim[0]
        tree_item_name = srh_sim_tree_item.name
        proj_file_name = os.environ.get('XMS_PYTHON_APP_PROJECT_PATH')
        proj_name = os.path.splitext(os.path.basename(proj_file_name))[0]
        proj_path = os.path.dirname(proj_file_name)
        sim_dir = os.path.join(proj_path, f'{proj_name}_models', 'SRH-2D', tree_item_name)
        if not os.path.isdir(sim_dir):
            sim_dir = os.path.join(proj_path, f'{proj_name}', 'SRH-2D', tree_item_name)
        srh_sim_item = self._coupler_data.srh_sim[1]
        sim_component = SrhSimComponent(srh_sim_item.main_file)
        case_name = sim_component.data.hydro.case_name
        monitor_data = {}
        for arc in arcs:
            arc_file = f'{case_name}_LN{arc.id}.dat'
            arc_file = os.path.join(sim_dir, 'Output_MISC', arc_file)
            file_data = np_arrays_from_file(arc_file)
            comp_id = self._monitor_component.get_comp_id(TargetType.arc, arc.id)
            if comp_id and comp_id > 0:
                monitor_data_param = self._monitor_component.data.monitor_arc_param_from_id(comp_id)
                if monitor_data_param.label:
                    monitor_data[monitor_data_param.label] = file_data
        points = self._node_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
        node_coverage_parameters = get_swmm_model().point_parameters
        node_dict = get_node_groups(points, self._node_component, node_coverage_parameters)
        diff_results = {}
        for index, row in enumerate(self._table):
            if row[0] in node_dict:
                value = node_dict[row[0]]
                if len(value) > 0:
                    point_id = value[0][2]
                    comp_id = self._node_component.get_comp_id(TargetType.point, point_id)
                    monitor_name = self._table[index][2]
                    if monitor_name in monitor_data:
                        cur_monitor_data = monitor_data[monitor_name]
                        times = cur_monitor_data[1][0] * 60
                        if len(times) > 0:
                            # Since the initial time in the Monitor line file is not 0.0, I want to add a time of
                            # 0.0 to the beginning of the array to give SWMM a value at time 0.0.
                            times = np.insert(times, 0, 0.0)
                        flows = cur_monitor_data[1][1]
                        if len(flows) > 0:
                            # Since the initial time in the Monitor line file is not 0.0, I want to repeat the starting
                            # flow value at the beginning of the array to give SWMM a value at time 0.0.
                            flows = np.insert(flows, 0, flows[0])
                        new_inflows = [list(times), list(flows)]
                        values = self._node_component.data.feature_values(TargetType.point, comp_id)
                        node_coverage_parameters.restore_values(values)
                        current_inflows = node_coverage_parameters.group('junction').parameter('inflows').value.copy()
                        difference = []
                        if len(current_inflows[1]) == len(new_inflows[1]):
                            difference = [new - current for new, current in zip(new_inflows[1], current_inflows[1])]
                        if difference:
                            new_inflows[1] = [(current + diff) * self._damping_factor for current, diff in
                                              zip(current_inflows[1], difference)]
                        node_coverage_parameters.group('junction').parameter('inflows').value = new_inflows
                        self._node_component.data.update_feature(TargetType.point,
                                                                 node_coverage_parameters.extract_values(), comp_id)
                        # Make comparison between existing and simulated inlet inflows
                        new_pd_inflows = pd.DataFrame({'min': new_inflows[0], 'vol_per_sec': new_inflows[1]})
                        current_pd_inflows = pd.DataFrame(
                            {'min': current_inflows[0], 'vol_per_sec': current_inflows[1]})
                        make_comparison(current_pd_inflows, new_pd_inflows, diff_results, self._table[index][0],
                                        'Time (minutes)', 'min')
        if diff_results:
            dlg = DifferenceViewer(diff_results, f'{swmm_sim_tree_item.name} Inlet Flow Change Info',
                                   'Inlet')
            dlg.exec()
        self._node_component.data.commit()
