"""Imports SRH Hydro file."""
# 1. Standard python modules
import logging

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg
from xms.srh.components.import_simulation_runner import SRHSimImporter

# 4. Local modules


__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"


class HydroImportRunner(QThread):
    """Worker thread for the .srhwhydro import script."""
    processing_finished = Signal()

    def __init__(self):
        """Constructor."""
        super().__init__()
        self._logger = logging.getLogger('xms.srhw')
        self.reader = None

    def run(self):
        """Imports an SRH-W simulation."""
        try:
            self.reader = SRHSimImporter()
            self.reader.read()
        except:  # noqa - # pragma no cover - hard to test exceptions using QueryPlayback
            self._logger.exception('Error importing SRH-W simulation.')
        finally:
            self.processing_finished.emit()


def import_hydro_with_feedback():
    """Import an .srhwhydro file with the feedback dialog."""
    xms_dlg.ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
    win_cont = xms_dlg.get_parent_window_container(parent_hwnd)
    # Create the timer that keeps our Python dialog in the foreground of XMS.
    _ = win_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope  # noqa

    worker = HydroImportRunner()
    display_text = {
        'title': 'SRH-W Simulation Import',
        'working_prompt': 'Importing SRH-W simulation. Please wait...',
        'warning_prompt': 'Warning(s) encountered reading the SRH-W simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered reading the SRH-W simulation. Review log output for more details.',
        'success_prompt': 'Successfully imported SRH-W simulation',
        'note': '',
        'auto_load': 'Auto load simulation into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.srhw', worker, win_cont)
    feedback_dlg.exec()
    win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
    if not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.reader.send()  # Send data back to XMS
