"""SimComponent class."""
# 1. Standard python modules
import json
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.guipy.param import param_h5_io
from xms.srh.data.par import par_util
from xms.srh.data.par.data_io_string import DataIoStringParam

# 4. Local modules
from xms.srhw.components.srhw_component import SrhwComponent
from xms.srhw.data.model_control_srh_w import ModelControlSrhW
from xms.srhw.gui.model_control_dialog_w import ModelControlDialogW

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"


class SimComponent(SrhwComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SRH-W model simulation."""

    def __init__(self, main_file):
        """Initializes the data class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.data = ModelControlSrhW()
        if os.path.isfile(self.main_file):
            self.data.output.output_maximum_datasets = False  # for compatibility with old datasets
            tmp_data = DataIoStringParam()
            param_h5_io.read_from_h5_file(self.main_file, tmp_data)
            if tmp_data.text:
                pd = json.loads(tmp_data.text)
                par_util.orjson_dict_to_param_cls(pdict=pd, cls=self.data, df_handler=None)
            else:  # old file version
                param_h5_io.read_from_h5_file(self.main_file, self.data)
        else:
            self._write_data_to_main_file()
        self.class_name = 'SimComponent'
        self.module_name = 'xms.srhw.components.sim_component'
        #                    [(menu_text, menu_method)...]
        self.tree_commands = [
            ('Model Control...', 'open_model_control'),
        ]

    def _write_data_to_main_file(self):
        """Writes this component's data to its main file."""
        self.data.param.file_type.precedence = 100
        self.data.param.file_version.precedence = 101
        self.data.param.mesh_uuid.precedence = 102
        tmp_data = DataIoStringParam()
        pd = par_util.param_cls_to_orjson_dict(cls=self.data, df_handler=None, skip_negative_precedence=True)
        tmp_data.text = json.dumps(pd)
        param_h5_io.write_to_h5_file(self.main_file, tmp_data)

    def link_event(self, link_dict, lock_state):
        """This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict (:obj:`dict`): A dictionary with keys being UUIDs as strings representing the objects being
                linked into this component. The values of this dictionary are a list of strings of the parameter
                names of the "takes" from the XML that this is a part of.

            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files
                if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

            messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
            tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
            text.

            action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        messages = []
        actions = []
        for link_uuid, link_xml_params in link_dict.items():
            for xml_param in link_xml_params:
                if xml_param in ['ugrid']:
                    if self.data.mesh_uuid and self.data.mesh_uuid != link_uuid:
                        # Already have a linked UGrid
                        params = {'old_uuid': self.data.mesh_uuid}
                        action = ActionRequest(
                            main_file=self.main_file, modality='NO_DIALOG', class_name=self.class_name,
                            module_name=self.module_name, method_name='delete_old_domain', comp_uuid=self.uuid,
                            parameters=params
                        )
                        actions.append(action)
                    self.data.mesh_uuid = link_uuid
        self._write_data_to_main_file()
        return messages, actions

    def unlink_event(self, unlinks, lock_state):
        """This will be called when a coverage, or a ugrid, or another component is unlinked from this component.

        Args:
            unlinks (:obj:`list` of str): A list of UUIDs as strings representing the objects being unlinked.

            lock_state (:obj:`bool`): True if the the component is locked for editing. Do not change the files
                if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        if self.data.mesh_uuid in unlinks:
            self.data.mesh_uuid = ''
        self._write_data_to_main_file()
        return [], []

    def delete_old_domain(self, query, params):
        """Delete the existing linked domain when another is linked to the simulation.

        This is needed because we want to allow using a UGrid object as the domain. The xml has a
        take_ugrid parameter with a limit of 1. If we get a link event for the domain and we already
        have one, we need to remove the old.

        Args:
            query (:obj:`xms.api.dmi.Query.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        old_uuid = None
        sim_uuid = None
        if params and params[0]:
            old_uuid = params[0].get('old_uuid')
            # Need the UUID of the parent tree item, even though it is the hidden component that actually has the XML
            # take parameter.
            sim_uuid = query.parent_item_uuid()
        if old_uuid and sim_uuid:
            query.unlink_item(taker_uuid=sim_uuid, taken_uuid=old_uuid)
        return [], []

    def open_model_control(self, query, params, win_cont):
        """Opens the Model Control dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

            messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
            tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
            text.

            action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        dlg = ModelControlDialogW(win_cont, self, query.copy_project_tree(), query)
        if dlg.exec():
            self.data = dlg.data
            self._write_data_to_main_file()
        return [], []
