"""Manages writing the SRH-W control file."""
# 1. Standard python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.srh.file_io.hydro_model_control_writer import HydroModelControlWriter

# 4. Local modules

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"


class HydroWriter:
    """Writes SRH-W geom file."""

    def __init__(self, file_name, model_control, file_list, logger, main_file='', grid_units='GridUnit "FOOT"'):
        """Constructor.

        Args:
            file_name (:obj:`str`): Path to the output .srhwhydro file
            model_control (:obj:`ModelControl`): The simulation data
            file_list (:obj:`list[str]`): Additional files that need to be exported
            logger (:obj:`logging.Logger`): Logger to use
            main_file (:obj:`str`): Path to simulation component main file
            grid_units (:obj:`str`): The grid units
        """
        self._file_name = file_name
        self._model_control = model_control
        self._file_list = file_list
        self._num_xy_files_written = 0  # track the number of curves we have written
        self.logger = logger
        self.grid_units = grid_units

        # these are set outside of this class if the data exists
        self.materials_manning = None
        self.run_name = None

    def write(self, run_name=None):
        """Write the .srhwhydro file.

        Args:
            run_name (:obj:`str`): Name of the run if this is a parameter run model
        """
        self.run_name = run_name  # Non-empty subfolder run name if doing parameters
        with open(self._file_name, 'w') as file:
            file.write('SRHWHYDRO\n')
            self._write_model_control(file)
            self._write_materials(file)

    def _write_model_control(self, file):
        """Write the materials data to the hydro file.

        Args:
            file (:obj:`file`): The output hydro file
        """
        writer = HydroModelControlWriter(file, self._model_control, self._file_list, grid_units=self.grid_units,
                                         logger=self.logger)
        writer.write(self.run_name)

    def _write_materials(self, file):
        """Write the materials data to the hydro file.

        Args:
            file: The output hydro file stream
        """
        if self.materials_manning:
            for i in range(len(self.materials_manning)):
                if type(self.materials_manning[i]) is list:
                    curve_name = '//ManningsN 30'
                    value = f'"{self.write_xys(self.materials_manning[i], curve_name=curve_name)}"'
                else:
                    value = f'{self.materials_manning[i]}'
                file.write(f'ManningsN {i} {value}\n')

    def write_xys(self, xys_list, curve_name='Curve'):
        """Write the xys file and return the file name.

        Args:
            xys_list (:obj:`list`): The list of tuples (x, y)
            curve_name (:obj:`str`): Curve name or 'Rating Curve'. First line of file.

        Returns:
            (:obj:`str`): Path from the model export location to the xys file
        """
        path = os.sep.join(self._file_name.split(os.sep)[:-1])
        self._num_xy_files_written += 1
        filename = os.path.join(path, f'curve_{self._num_xy_files_written}.xys')
        with open(filename, 'w') as file:
            file.write(f'{curve_name} {self._num_xy_files_written}\n')
            for item in xys_list:
                file.write(f'{item[0]} {item[1]}\n')
        return os.path.basename(filename)
