"""Map locations and attributes of all linked coverages to the SRH-2D domain."""
# 1. Standard python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.srh.mapping.bc_mapper import BcMapper

# 4. Local modules

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"


class CoverageMapper:
    """Class for mapping coverages to a mesh for SRH-2D."""
    def __init__(self, query_helper):
        """Constructor."""
        super().__init__()
        self._logger = logging.getLogger('xms.srh')

        self.grid_units = query_helper.grid_units
        self.grid_name = query_helper.grid_name
        self.co_grid = query_helper.co_grid
        self.grid_uuid = query_helper.grid_uuid
        self.grid_wkt = query_helper.grid_wkt
        self.component_folder = query_helper.component_folder

        bc_cov = query_helper.coverages.get('Boundary Conditions', [None, None])
        self.bc_coverage = bc_cov[0]
        self.bc_component_file = bc_cov[1]
        self.bc_component = query_helper.bc_component
        self.bc_arc_id_to_grid_ids = None
        self.bc_arc_id_to_comp_id = None
        self.bc_arc_id_to_bc_param = None
        self.bc_arc_id_to_bc_id = None
        self.bc_id_to_structures = None
        self.bc_hy8_file = None
        self.bc_3d_structures = []
        self.bc_mapped_comp_uuid = None
        self.bc_mapped_comp_display_uuid = None

        self.mapped_comps = []

    def do_map(self):
        """Creates the snap preview of coverages onto the mesh."""
        try:
            self._map_bc()
        except:  # pragma: no cover  # noqa
            self._logger.exception('Error generating snap.')  # pragma: no cover

    def _map_bc(self):
        """Maps the bcs from the bc coverage to the mesh."""
        if self.bc_coverage is None or not self.bc_component_file:
            return
        self._logger.info('Mapping bc coverage to mesh.')
        mapper = BcMapper(self, wkt=self.grid_wkt, generate_snap=False)
        mapper.bc_mapped_comp_uuid = self.bc_mapped_comp_uuid
        mapper.bc_mapped_comp_display_uuid = self.bc_mapped_comp_display_uuid
        self.bc_arc_id_to_grid_ids = mapper.arc_id_to_grid_ids
        self.bc_arc_id_to_comp_id = mapper._arc_id_to_comp_id
        self.bc_arc_id_to_bc_param = mapper._arc_id_to_bc_param
        self.bc_arc_id_to_bc_id = mapper._arc_id_to_bc_id
        self.bc_id_to_structures = mapper._structures
        self.bc_hy8_file = mapper._bc_component.hy8_file
        self.bc_3d_structures = mapper.out_3d_structures
        mapper.structures_3d_weirs = []
        do_comp, comp = mapper.do_map()
        if do_comp is not None:
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()],
                                     'mapped_bc_component'))
        self._logger.info('Finished mapping bc coverage to mesh.')
