"""Simulation component class."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from packaging.version import parse as parse_version

# 3. Aquaveo modules
from xms.constraint import read_grid_from_file
from xms.guipy.time_format import XmsTimeFormatter

# 4. Local modules
from xms.stwave.components.stwave_base_component import StwaveBaseComponent
from xms.stwave.data.simulation_data import SimulationData
from xms.stwave.gui.model_control_dialog import ModelControlDialog
from xms.stwave.gui.partial_export_dialog import PartialExportDialog


class SimulationComponent(StwaveBaseComponent):
    """A hidden Dynamic Model Interface (DMI) component for the STWAVE model simulation."""

    def __init__(self, main_file):
        """
        Initializes the data class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = SimulationData(main_file)
        #                    [(menu_text, menu_method)...]
        self.tree_commands = [
            ('Model Control...', 'open_model_control'),
            ('Export...', 'open_partial_export'),
        ]
        if not os.path.isfile(self.main_file):
            self.data.commit()

    def save_to_location(self, new_path, save_type):
        """
        Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        file_version = self.data.info.attrs['VERSION']
        if parse_version(file_version) < parse_version('2.2.0'):
            if os.environ.get('XMS_OPENING_PROJECT', '') == 'TRUE':
                commit = False
                if self.data.info.attrs['monitoring_uuid'] != '':
                    self.data.info.attrs['monitoring'] = 1
                    commit = True
                if self.data.info.attrs['nesting_uuid'] != '':
                    self.data.info.attrs['nesting'] = 1
                    commit = True
                if commit:
                    self.data.commit()
        return new_main_file, messages, action_requests

    def open_model_control(self, query, params, win_cont):
        """
        Opens the dialog and saves component data state on OK.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS.
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            tuple[list, list]:
                - messages (list[tuple[str, str]): List of tuples with the first element of the tuple being the message
                  level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
                - action_requests (list[xms.api.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        spec_cov = query.item_with_uuid(self.data.info.attrs['spectral_uuid'], generic_coverage=True)
        grid_data = {}
        do_ugrid = query.item_with_uuid(self.data.info.attrs['grid_uuid'])
        if do_ugrid is not None:
            grid_data['cogrid'] = read_grid_from_file(do_ugrid.cogrid_file)
            grid_data['grid_name'] = do_ugrid.name
        time_formatter = XmsTimeFormatter(query.global_time_settings)
        dialog = ModelControlDialog(data=self.data, pe_tree=query.project_tree, spec_cov=spec_cov,
                                    time_format=time_formatter.qt_abs_specifier, grid_data=grid_data, parent=win_cont)
        if dialog.exec_():
            self.data.commit()
        return [], []

    def open_partial_export(self, query, params, win_cont):
        """Opens the dialog for exporting individual files of an STWAVE simulation.

        Args:
            query (xms.api.dmi.Query): Object for communicating with XMS
            params (dict): Generic map of parameters. Unused in this case.
            win_cont (PySide2.QtWidgets.QWidget): The window container.

        Returns:
            tuple[list, list]:
                - messages (list[tuple[str, str]): List of tuples with the first element of the tuple being the message
                  level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
                - action_requests (list[xms.api.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        dlg = PartialExportDialog(win_cont, query)
        dlg.exec()
        return [], []

    def link_event(self, link_dict, lock_state):
        """This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict (dict): A dictionary with keys being UUIDs as strings representing the objects being linked into
                this component. The values of this dictionary are a list of strings of the parameter names of the
                "takes" from the XML that this is a part of.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            tuple[list, list]:
                - messages (list[tuple[str, str]): List of tuples with the first element of the tuple being the message
                  level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
                - action_requests (list[xms.api.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        messages = []
        actions = []
        for link_uuid, link_xml_params in link_dict.items():
            for xml_param in link_xml_params:
                if xml_param == 'stwGrid':
                    self.data.info.attrs['grid_uuid'] = link_uuid
                elif xml_param == 'CovSpectral':
                    self.data.info.attrs['spectral_uuid'] = link_uuid
        self.data.commit()
        return messages, actions

    def unlink_event(self, unlinks, lock_state):
        """This will be called when a coverage, or a ugrid, or another component is unlinked from this component.

        Args:
            unlinks (:obj:`list` of str): A list of UUIDs as strings representing the objects being unlinked.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            tuple[list, list]:
                - messages (list[tuple[str, str]): List of tuples with the first element of the tuple being the message
                  level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
                - action_requests (list[xms.api.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        if self.data.info.attrs['grid_uuid'] in unlinks:
            self.data.info.attrs['grid_uuid'] = ''
        if self.data.info.attrs['spectral_uuid'] in unlinks:
            self.data.info.attrs['spectral_uuid'] = ''
        self.data.commit()
        return [], []
