"""Worker thread for the simulation importer."""

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules


class SimImportWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing a CMS-Wave .sim control file."""

    def __init__(self, parent):
        """Construct the worker.

        Args:
            parent (QWidget): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._reader = None

    def _do_work(self):
        """Thread runner that imports the fort.15."""
        try:
            from xms.stwave.file_io.sim_reader import SimReader
            self._reader = SimReader()
            self._reader.read()
        except Exception:
            logging.getLogger('xms.stwave').exception('Error(s) encountered while reading STWAVE files.')

    def send(self):
        """Send data to XMS if import is successful."""
        if self._reader and self._reader.query:
            self._reader.query.send()


def import_sim_with_feedback():
    """Driver for the .sim import script with a feedback dialog."""
    ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    if XmEnv.xms_environ_running_tests() != 'TRUE':
        parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
        parent = xms_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = win_gui.create_and_connect_raise_timer(main_hwnd, parent)  # Keep the timer in scope
    else:
        parent = None
        main_hwnd = None

    worker = SimImportWorkerThread(parent)
    display_text = {
        'title': 'Reading STWAVE Simulation',
        'working_prompt': 'Reading STWAVE sim file. Please wait...',
        'error_prompt': 'Error(s) encountered reading STWAVE simulation. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered reading STWAVE simulation. Review log output for more details.',
        'success_prompt': 'Successfully read STWAVE simulation.',
        'note': '',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, parent)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec() and not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.send()  # Send the data to SMS
    if main_hwnd is not None:
        win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
