"""A reader for STWAVE simulation files."""

# 1. Standard Python modules
import datetime
import logging
import os
import uuid

# 2. Third party modules
import numpy as np
import pandas

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.constraint.rectilinear_geometry import Numbering, Orientation
from xms.constraint.rectilinear_grid_builder import RectilinearGridBuilder
from xms.core.filesystem import filesystem as xfs
from xms.data_objects.parameters import Component, Coverage, Point, Projection, Simulation, UGrid
from xms.datasets import dataset_reader
from xms.guipy.time_format import ISO_DATETIME_FORMAT

# 4. Local modules
from xms.stwave.data import simulation_data
from xms.stwave.data import stwave_consts as const
from xms.stwave.data.simulation_data import SimulationData
from xms.stwave.file_io.dataset_reader import DatasetReaderSTWAVE
from xms.stwave.file_io.eng_reader import EngReader, get_coords_from_ij


def snap_idd_to_datetime(snap_idd):
    """Parse a datetime from a snap idd line.

    Args:
        snap_idd (str): The snap idd line to parse

    Returns:
        datetime.datetime: The parsed datetime or Jan 1, 1950 if invalid format
    """
    try:
        year = int(snap_idd[:4])
        month = int(snap_idd[4:6])
        day = int(snap_idd[6:8])
        hour = int(snap_idd[8:10]) if len(snap_idd) >= 10 else 0
        minute = int(snap_idd[10:12]) if len(snap_idd) >= 12 else 0
        second = int(snap_idd[12:]) if len(snap_idd) == 14 else 0
        return datetime.datetime(year, month, day, hour, minute, second)
    except Exception:
        return datetime.datetime(year=1950, month=1, day=1)


class SimReader:
    """A class for reading STWAVE simulation files."""

    def __init__(self, filename='', query=None):
        """Constructor.

        Args:
            filename (str): Used for testing. Avoids having to get the sim filename from the Query.
            query (Query): XMS interprocess communicator
        """
        self._logger = logging.getLogger('xms.stwave')
        self._reftime = datetime.datetime(year=1950, month=1, day=1)
        self.filename = filename
        self.query = query
        self.using_jonswap = False
        self.transient_depth = False
        self.grid_z = None
        self.grid_proj = Projection()
        self.grid_data = {}  # {card: value}
        self.geom_uuid = ''
        self.input_files = {}  # {dset_name: [filename, format]}
        self.case_times = []
        self.idd_cards = {}
        self.snap_idds = []
        self.select_pts = []
        self.nest_pts = []
        self.station_pts = []
        self.subset_pts = []
        self.const_mag = []
        self.const_dir = []
        self.const_surge = []
        # self.debugger = open('debug.txt', 'w')
        self.comp_dir = ''
        self.sim_comp_dir = ''
        self.sim_data = None
        self.sim_uuid = ''
        self.do_sim = None
        self.do_sim_comp = None
        self.setup_query()

    def setup_query(self):
        """Sets up simulation component."""
        if self.filename and not self.query:
            # If the sim filename has already been provided, we are in a test. We can't rely on the environment
            # variable here because recording tests need to hit that code. For non-recording tests, we can just
            # supply the filename to avoid the Query.
            return
        if self.query is None:  # Allow a Query to be passed in from other scripts
            self.query = Query()
        self.comp_dir = os.path.join(self.query.xms_temp_directory, 'Components')
        comp = Component()
        comp_uuid = str(uuid.uuid4())
        self.sim_comp_dir = os.path.join(self.comp_dir, comp_uuid)
        os.makedirs(self.sim_comp_dir, exist_ok=True)
        sim_mainfile = os.path.join(self.sim_comp_dir, 'simulation_comp.nc')
        self.sim_data = SimulationData(sim_mainfile)  # This will create a default model control
        comp.uuid = comp_uuid
        comp.name = 'sim_comp'
        comp.main_file = sim_mainfile
        comp.set_unique_name_and_model_name('Sim_Component', 'STWAVE')
        comp.locked = False

        # Build the STWAVE simulation
        sim = Simulation()
        sim.model = 'STWAVE'
        sim.uuid = str(uuid.uuid4())
        self.sim_uuid = sim.uuid
        self.do_sim = sim
        self.do_sim_comp = comp

    def get_case_times(self):  # for idd_spec_types without snap_idds
        """
        Gets case times for 'none' key.

        Returns:
            (:obj:`list` of :obj:`datetime.datetime`): List of times.
        """
        ts_times = []
        for case_time in self.case_times:
            ts_times.append(case_time)
        return ts_times

    def get_idd_to_case_time_map(self):  # for idd_spec_types with snap_idds
        """Gets the case times and corresponding snap idd mapped together.

        Returns:
            (:obj:`dict` of :obj:`datetime.datetime`): snap idd : case time
        """
        idd_to_time = {}
        for idx, snap_idd in enumerate(self.snap_idds):
            idd_to_time[snap_idd] = self.case_times[idx]
        return idd_to_time

    def read_all_cards(self):
        """Reads all the cards in file and loads data members with the info.

        Widgets, grid data, input files, idd cards, Points and others are set.

        Returns:
            (str): Name of file used.
        """
        if not self.filename:
            self.filename = self.query.read_file

        with open(self.filename, 'r') as in_file:
            for line in in_file:
                line = line.strip().replace('\'', '')
                if not line or line[0] == '#' or line[0] == '&' or line[0] == '/':
                    continue

                line_data = [x.strip().replace('"', '') for x in line.split('=') if x != '']
                # remove comma at end of the line
                line_data[len(line_data) - 1] = line_data[len(line_data) - 1].replace(',', '')
                token = line_data[0].lower()

                # std_parms namelist
                if token == 'iplane':
                    option = const.PLANE_TYPE_FULL
                    if line_data[1] == '0':
                        option = const.PLANE_TYPE_HALF
                    self.sim_data.info.attrs['plane'] = option
                elif token == 'iprp':
                    option = const.SOURCE_PROP_AND_TERMS
                    if line_data[1] == '1':
                        option = const.SOURCE_PROP_ONLY
                    self.sim_data.info.attrs['source_terms'] = option
                elif token == 'icur':
                    option = const.OPT_NONE
                    if line_data[1] == '1':
                        option = const.OPT_DSET
                    self.sim_data.info.attrs['current_interaction'] = option
                elif token == 'ibreak':
                    option = const.BREAK_OPT_NONE
                    if line_data[1] == '1':
                        option = const.BREAK_OPT_WRITE
                    elif line_data[1] == '2':
                        option = const.BREAK_OPT_CALCULATE
                    self.sim_data.info.attrs['breaking_type'] = option
                elif token == 'irs':
                    self.sim_data.info.attrs['rad_stress'] = int(line_data[1])
                elif token == 'ibnd':
                    option = const.INTERP_OPT_NONE
                    if line_data[1] == '1':
                        option = const.INTERP_OPT_LINEAR
                    elif line_data[1] == '2':
                        option = const.INTERP_OPT_MORPHIC
                    self.sim_data.info.attrs['interpolation'] = option
                elif token == 'ifric':
                    option = const.OPT_NONE
                    if line_data[1] == '1':
                        option = const.FRIC_OPT_JONSWAP_CONST
                        self.using_jonswap = True
                    elif line_data[1] == '2':
                        option = const.FRIC_OPT_JONSWAP_DSET
                        self.using_jonswap = True
                    elif line_data[1] == '3':
                        option = const.FRIC_OPT_MANNING_CONST
                    elif line_data[1] == '4':
                        option = const.FRIC_OPT_MANNING_DSET
                    self.sim_data.info.attrs['friction'] = option
                elif token == 'isurge':
                    option = const.OPT_CONST
                    if line_data[1] == '1':
                        option = const.OPT_DSET
                    self.sim_data.info.attrs['surge'] = option
                elif token == 'iwind':
                    option = const.OPT_CONST
                    if line_data[1] == '1':
                        option = const.OPT_DSET
                    self.sim_data.info.attrs['wind'] = option
                elif token == 'idep_opt':
                    option = const.DEP_OPT_NONTRANSIENT
                    if line_data[1] == '2':
                        option = const.DEP_OPT_TRANSIENT
                        self.transient_depth = True
                    elif line_data[1] == '3':
                        option = const.DEP_OPT_COUPLED
                    self.sim_data.info.attrs['depth'] = option
                elif 'i_bc' in token:
                    option = const.I_BC_ZERO
                    if line_data[1] == '2':
                        option = const.I_BC_SPECIFIED
                    elif line_data[1] == '3':
                        option = const.I_BC_LATERAL
                    attr_name = 'side1'
                    if '2' in token:
                        attr_name = 'side2'
                    elif '3' in token:
                        attr_name = 'side3'
                    elif '4' in token:
                        attr_name = 'side4'
                    self.sim_data.info.attrs[attr_name] = option
                elif token == 'iice':
                    option = const.OPT_NONE
                    if line_data[1] == '1':
                        option = const.OPT_DSET
                    self.sim_data.info.attrs['ice'] = option
                elif token == 'percent_ice_threshold':
                    self.sim_data.info.attrs['ice_threshold'] = float(line_data[1])
                # END std_parms namelist
                # run_parms namelist
                elif token == 'numsteps':
                    self.idd_cards['numsteps'] = int(line_data[1])
                elif token == 'idd_spec_type':
                    self.idd_cards['idd_spec_type'] = int(line_data[1])
                elif token == 'n_grd_part_i':
                    self.sim_data.info.attrs['processors_i'] = int(line_data[1])
                elif token == 'n_grd_part_j':
                    self.sim_data.info.attrs['processors_j'] = int(line_data[1])
                elif token == 'n_init_iters':
                    self.sim_data.info.attrs['max_init_iters'] = int(line_data[1])
                elif token == 'init_iters_stop_value':
                    self.sim_data.info.attrs['init_iters_stop_value'] = float(line_data[1])
                elif token == 'init_iters_stop_percent':
                    self.sim_data.info.attrs['init_iters_stop_percent'] = float(line_data[1])
                elif token == 'n_final_iters':
                    self.sim_data.info.attrs['max_final_iters'] = int(line_data[1])
                elif token == 'final_iters_stop_value':
                    self.sim_data.info.attrs['final_iters_stop_value'] = float(line_data[1])
                elif token == 'final_iters_stop_percent':
                    self.sim_data.info.attrs['final_iters_stop_percent'] = float(line_data[1])
                # END run_parms namelist
                # spatial_grid_parms namelist
                elif token == 'coord_sys':
                    coord_sys = 'LOCAL'
                    if line_data[1].upper() == 'STATEPLANE':
                        coord_sys = 'STATEPLANE'
                    elif line_data[1].upper() == 'UTM':
                        coord_sys = 'UTM'
                    self.grid_data['coord_sys'] = coord_sys
                elif token == 'spzone':
                    self.grid_data['spzone'] = int(line_data[1])
                elif token == 'x0':
                    self.grid_data['x0'] = float(line_data[1])
                elif token == 'y0':
                    self.grid_data['y0'] = float(line_data[1])
                elif token == 'azimuth':
                    self.grid_data['azimuth'] = float(line_data[1])
                elif token == 'dx':
                    self.grid_data['dx'] = float(line_data[1])
                elif token == 'dy':
                    self.grid_data['dy'] = float(line_data[1])
                elif token == 'n_cell_i':
                    self.grid_data['n_cell_i'] = int(line_data[1])
                elif token == 'n_cell_j':
                    self.grid_data['n_cell_j'] = int(line_data[1])
                # END spatial_grid_parms namelist
                # input_files namelist
                elif token == 'dep':
                    if self.transient_depth:
                        self.input_files['Transient Depth'] = [line_data[1], 1]
                    else:
                        self.input_files['Depth'] = [line_data[1], 1]
                elif token == 'surge':
                    self.input_files['Surge'] = [line_data[1], 1]
                elif token == 'spec':
                    self.input_files['spec'] = [line_data[1], 1]
                elif token == 'wind':
                    self.input_files['Wind'] = [line_data[1], 1]
                elif token == 'fric':
                    self.input_files['Friction'] = [line_data[1], 1]  # JONSWAP or Manning's N
                elif token == 'curr':
                    self.input_files['Current'] = [line_data[1], 1]
                elif token == 'ice':
                    self.input_files['Ice'] = [line_data[1], 1]
                elif token == 'io_type_dep':
                    if 'Transient Depth' in self.input_files:
                        self.input_files['Transient Depth'][1] = int(line_data[1])
                    elif 'Depth' in self.input_files:
                        self.input_files['Depth'][1] = int(line_data[1])
                elif token == 'io_type_surge':
                    if 'Surge' in self.input_files:
                        self.input_files['Surge'][1] = int(line_data[1])
                elif token == 'io_type_wind':
                    if 'Wind' in self.input_files:
                        self.input_files['Wind'][1] = int(line_data[1])
                elif token == 'io_type_spec':
                    if 'spec' in self.input_files:
                        self.input_files['spec'][1] = int(line_data[1])
                elif token == 'io_type_fric':
                    if 'Friction' in self.input_files:
                        self.input_files['Friction'][1] = int(line_data[1])
                elif token == 'io_type_curr':
                    if 'Current' in self.input_files:
                        self.input_files['Current'][1] = int(line_data[1])
                elif token == 'io_type_ice':
                    if 'Ice' in self.input_files:
                        self.input_files['Ice'][1] = int(line_data[1])
                # END input_files namelist
                # time_parms namelist
                elif token == 'i_time_inc':
                    self.idd_cards['i_time_inc'] = int(line_data[1])
                elif token == 'i_time_inc_units':
                    self.idd_cards['i_time_inc_units'] = line_data[1]
                elif token == 'iyear_start':
                    self.idd_cards['iyear_start'] = int(line_data[1])
                elif token == 'imon_start':
                    self.idd_cards['imon_start'] = int(line_data[1])
                elif token == 'iday_start':
                    self.idd_cards['iday_start'] = int(line_data[1])
                elif token == 'ihr_start':
                    self.idd_cards['ihr_start'] = int(line_data[1])
                elif token == 'imin_start':
                    self.idd_cards['imin_start'] = int(line_data[1])
                elif token == 'isec_start':
                    self.idd_cards['isec_start'] = int(line_data[1])
                elif token == 'iyear_end':
                    self.idd_cards['iyear_end'] = int(line_data[1])
                elif token == 'imon_end':
                    self.idd_cards['imon_end'] = int(line_data[1])
                elif token == 'iday_end':
                    self.idd_cards['iday_end'] = int(line_data[1])
                elif token == 'ihr_end':
                    self.idd_cards['ihr_end'] = int(line_data[1])
                elif token == 'imin_end':
                    self.idd_cards['imin_end'] = int(line_data[1])
                elif token == 'isec_end':
                    self.idd_cards['isec_end'] = int(line_data[1])
                # END time_parms namelist
                # const_spec namelist
                elif token == 'nfreq':
                    self.sim_data.info.attrs['num_frequencies'] = int(line_data[1])
                elif token == 'fa' or token == 'f0':
                    self.sim_data.info.attrs['min_frequency'] = float(line_data[1])
                elif token == 'df_const':
                    self.sim_data.info.attrs['delta_frequency'] = float(line_data[1])
                # END const_spec namelist
                # const_fric namelist
                elif token == 'cf_const':
                    if self.using_jonswap:
                        self.sim_data.info.attrs['JONSWAP'] = float(line_data[1])
                    else:
                        self.sim_data.info.attrs['manning'] = float(line_data[1])
                # END const_fric namelist
                # snap_idds namelist
                elif token.startswith('idds('):
                    self.snap_idds.append(line_data[1].strip())
                # END snap_idds namelist
                # select_pts namelist
                elif token.startswith('iout('):
                    ival = float(line_data[1].split(',')[0].strip())  # strip away non-numericals at end of string
                    jval = float(line_data[2])
                    ptx, pty = get_coords_from_ij(ival, jval, self.grid_data['azimuth'], self.grid_data['x0'],
                                                  self.grid_data['y0'], self.grid_data['dx'], self.grid_data['dy'])
                    select_pt = Point(float(ptx), float(pty))
                    select_pt.id = len(self.select_pts) + 1
                    self.select_pts.append(select_pt)
                # END select_pts namelist
                # nest_pts namelist
                elif token.startswith('inest('):
                    ival = float(line_data[1].split(',')[0].strip())  # strip away non-numericals at end of string
                    jval = float(line_data[2])
                    ptx, pty = get_coords_from_ij(ival, jval, self.grid_data['azimuth'], self.grid_data['x0'],
                                                  self.grid_data['y0'], self.grid_data['dx'], self.grid_data['dy'])
                    nest_pt = Point(float(ptx), float(pty))
                    nest_pt.id = len(self.nest_pts) + 1
                    self.nest_pts.append(nest_pt)
                # station_locations namelist
                elif token.startswith('stat_xcoor('):
                    ptx = line_data[1].split(',')[0].strip()  # strip away non-numericals at end of string
                    station_pt = Point(float(ptx), float(line_data[2]))
                    station_pt.id = len(self.station_pts) + 1
                    self.station_pts.append(station_pt)
                # END station_locations namelist
                # subsample_speceng namelist
                elif token.startswith('xloc_subsample('):
                    ptx = line_data[1].split(',')[0].strip()  # strip away non-numericals at end of string
                    subset_pt = Point(float(ptx), float(line_data[2]))
                    subset_pt.id = len(self.subset_pts) + 1
                    self.subset_pts.append(subset_pt)
                # END subsample_speceng namelist
                # const_wind namelist
                elif token.startswith('umag_const_in('):
                    mag = line_data[1].split(',')[0].strip()  # strip away non-numericals at end of string
                    self.const_mag.append(float(mag))
                    self.const_dir.append(float(line_data[2]))
                # END const_wind namelist
                # const_surge namelist
                elif token.startswith('dadd_const_in('):
                    self.const_surge.append(float(line_data[1]))
                # END const_surge namelist

        return self.filename

    def build_grid(self, name):
        """
        Builds a Rectilinear Grid from grid data members.

        Args:
            name (str): Name of grid.
        """
        self._logger.info('Building grid...')
        builder = RectilinearGridBuilder()
        builder.angle = self.grid_data['azimuth']
        builder.origin = (self.grid_data['x0'], self.grid_data['y0'], 0.0)
        builder.numbering = Numbering.kji
        builder.orientation = (Orientation.x_increase, Orientation.y_increase)
        builder.is_2d_grid = True
        builder.is_3d_grid = False
        if self.grid_data['dx'] == self.grid_data['dy']:
            builder.set_square_xy_locations(self.grid_data['n_cell_i'] + 1, self.grid_data['n_cell_j'] + 1,
                                            self.grid_data['dx'])
        else:
            builder.locations_x = [self.grid_data['dx'] * i for i in range(self.grid_data['n_cell_i'] + 1)]
            builder.locations_y = [self.grid_data['dy'] * j for j in range(self.grid_data['n_cell_j'] + 1)]
        rect_grid = builder.build_grid()
        if self.grid_z:
            rect_grid.cell_elevations = self.grid_z  # already in elevations
        cogrid_file = os.path.join(self.sim_comp_dir, 'stwave_domain.xmc')
        rect_grid.write_to_file(cogrid_file, True)
        ugrid = UGrid(cogrid_file, name=name)
        ugrid.projection = self.grid_proj
        self.geom_uuid = str(uuid.uuid4())
        ugrid.uuid = self.geom_uuid
        self.sim_data.info.attrs['grid_uuid'] = self.geom_uuid
        self.query.add_ugrid(ugrid)
        self.query.link_item(self.sim_uuid, self.geom_uuid)

    def build_coverages(self):
        """
        Builds coverages from points and grid members.

        Monitor, nesting points, output, and subset coverages are all built
        and loaded.
        """
        # build the monitor coverage
        if self.select_pts:
            self._logger.info('Creating monitoring coverage...')
            monitor_cov = Coverage()
            monitor_cov.name = 'STWAVE Monitoring Cells'
            monitor_cov.set_points(self.select_pts)
            monitor_cov.projection = self.grid_proj
            monitor_cov.uuid = str(uuid.uuid4())
            self.sim_data.info.attrs['monitoring'] = 1
            self.sim_data.info.attrs['monitoring_uuid'] = monitor_cov.uuid
            monitor_cov.complete()
            self.query.add_coverage(monitor_cov)

        # build the nesting points coverage
        if self.nest_pts:
            self._logger.info('Creating nesting coverage...')
            nest_cov = Coverage()
            nest_cov.name = 'STWAVE Nesting Points'
            nest_cov.set_points(self.nest_pts)
            nest_cov.projection = self.grid_proj
            nest_cov.uuid = str(uuid.uuid4())
            self.sim_data.info.attrs['nesting'] = 1
            self.sim_data.info.attrs['nesting_uuid'] = nest_cov.uuid
            nest_cov.complete()
            self.query.add_coverage(nest_cov)

        if self.station_pts:
            self._logger.info('Creating station points coverage...')
            output_cov = Coverage()
            output_cov.name = 'STWAVE Station Points'
            output_cov.set_points(self.station_pts)
            output_cov.projection = self.grid_proj
            output_cov.uuid = str(uuid.uuid4()).encode('ascii', 'ignore')
            output_cov.complete()
            self.sim_data.info.attrs['output_stations'] = 1
            self.sim_data.info.attrs['output_stations_uuid'] = output_cov.uuid
            self.query.add_coverage(output_cov)

        if self.subset_pts:
            self._logger.info('Creating spectral subset coverage...')
            subset_cov = Coverage()
            subset_cov.name = 'STWAVE Spectral Subset'
            subset_cov.set_points(self.subset_pts)
            subset_cov.projection = self.grid_proj
            subset_cov.uuid = str(uuid.uuid4()).encode('ascii', 'ignore')
            subset_cov.complete()
            self.sim_data.info.attrs['location_coverage'] = 1
            self.sim_data.info.attrs['location_coverage_uuid'] = subset_cov.uuid
            self.query.add_coverage(subset_cov)

    def get_case_times_for_context(self):
        """Updates case_times with data relative to reftime."""
        # Try setting the reference time from the first snap idd
        if self.idd_cards['idd_spec_type'] == -2 and self.snap_idds:
            self._reftime = snap_idd_to_datetime(self.snap_idds[0])
        year = self._reftime.year
        if 'iyear_start' in self.idd_cards:
            year = int(self.idd_cards['iyear_start'])
        month = self._reftime.month
        if 'imon_start' in self.idd_cards:
            month = int(self.idd_cards['imon_start'])
        day = self._reftime.day
        if 'iday_start' in self.idd_cards:
            day = int(self.idd_cards['iday_start'])
        hour = 0
        if 'ihr_start' in self.idd_cards:
            hour = int(self.idd_cards['ihr_start'])
        minute = 0
        if 'imin_start' in self.idd_cards:
            minute = int(self.idd_cards['imin_start'])
        second = 0
        if 'isec_start' in self.idd_cards:
            second = int(self.idd_cards['isec_start'])
        self._reftime = datetime.datetime(year, month, day, hour, minute, second)
        self.sim_data.info.attrs['reftime'] = self._reftime.strftime(ISO_DATETIME_FORMAT)

        # get the time units if provided, otherwise use hours
        multiplier = 60 * 60
        if 'i_time_inc_units' in self.idd_cards:
            time_units = self.idd_cards['i_time_inc_units'].lower()
            if time_units == 'ss':
                multiplier = 1
            elif time_units == 'mm':
                multiplier = 60
            elif time_units == 'dd':
                multiplier = 60 * 60 * 24

        # get constant time increment if provided
        increment = None
        if 'i_time_inc' in self.idd_cards:
            increment = int(self.idd_cards['i_time_inc'])

        # get the number of timesteps if provided, otherwise get the end date
        numsteps = None  # required except when idd_spec_type == 2
        if 'numsteps' in self.idd_cards:
            numsteps = self.idd_cards['numsteps']
        else:  # get the end date which is required now (idd_spec_type == 2)
            end_year = self.idd_cards['iyear_end']
            end_month = self.idd_cards['imon_end']
            end_day = self.idd_cards['iday_end']
            end_hour = self.idd_cards['ihr_end']
            end_min = self.idd_cards['imin_end']
            end_sec = self.idd_cards['isec_end']
            end_date = datetime.datetime(end_year, end_month, end_day, end_hour, end_min, end_sec)
            ts_time = self._reftime
            while ts_time < end_date:
                numsteps += 1
                ts_time = ts_time + datetime.timedelta(seconds=increment * multiplier)
                self.case_times.append(ts_time)

        if self.idd_cards['idd_spec_type'] == -2:  # absolute timestamps (what SMS exports)
            for snap in self.snap_idds:
                ts_time = snap_idd_to_datetime(snap)
                self.case_times.append(ts_time)
        elif self.idd_cards['idd_spec_type'] == -3:  # relative to the reftime
            for snap in self.snap_idds:
                ts_offset = datetime.timedelta(seconds=int(snap) * multiplier)
                ts_time = self._reftime + ts_offset
                self.case_times.append(ts_time)
        else:
            for i in range(numsteps):
                step_size = i
                if increment:  # used specified constant interval if provided
                    step_size = increment * i
                ts_offset = datetime.timedelta(seconds=step_size * multiplier)
                ts_time = self._reftime + ts_offset
                self.case_times.append(ts_time)

    def build_spatial_datasets(self):
        """Builds the spatial datasets and adds it to query."""
        # read the input datasets
        self._logger.info('Importing spatial datasets...')
        ts_times = self.get_case_times()
        dsets = []
        for k, v in self.input_files.items():
            fname = xfs.resolve_relative_path(os.path.dirname(self.filename), v[0])
            # Datasets defined in the sim file do not actually have to exist, so skip them silently.
            if k != 'spec' and os.path.isfile(fname):  # spectral datasets are their own format and can of worms
                reftime = datetime.datetime.strptime(self.sim_data.info.attrs['reftime'], ISO_DATETIME_FORMAT)
                reader = DatasetReaderSTWAVE(k, fname, ts_times, self.geom_uuid, angle=self.grid_data['azimuth'],
                                             reftime=reftime)
                dsets.append(reader.read())
        for dset in dsets:
            if dset:
                name = dset.name
                is_z = False
                # push an empty node in for the widget
                if name == 'Depth':  # z-values on the grid
                    reader = dataset_reader.DatasetReader(h5_filename=dset.h5_filename, group_path=dset.group_path)
                    self.grid_z = reader.values[0].tolist()
                    is_z = True
                elif name == 'Transient Depth':
                    reader = dataset_reader.DatasetReader(h5_filename=dset.h5_filename, group_path=dset.group_path)
                    self.grid_z = reader.values[0].tolist()
                    self.sim_data.info.attrs['depth_uuid'] = dset.uuid
                elif name == 'Current':
                    self.sim_data.info.attrs['current_uuid'] = dset.uuid
                elif name == 'Friction':
                    if self.using_jonswap:
                        self.sim_data.info.attrs['JONSWAP_uuid'] = dset.uuid
                    else:
                        self.sim_data.info.attrs['manning_uuid'] = dset.uuid
                elif name == 'Surge':
                    self.sim_data.info.attrs['surge_uuid'] = dset.uuid
                elif name == 'Wind':
                    self.sim_data.info.attrs['wind_uuid'] = dset.uuid
                elif name == 'Ice':
                    self.sim_data.info.attrs['ice_uuid'] = dset.uuid
                else:
                    continue
                # add the dataset itself as a node in the context
                if not is_z:
                    self.query.add_dataset(dset)

    def read_spectral_cov(self):
        """Builds an energy reader, then reads the data and loads it into widgets."""
        self._logger.info('Reading spectral coverage...')
        idd_to_time = None
        simple_time = None
        if self.snap_idds:
            idd_to_time = self.get_idd_to_case_time_map()
        else:
            simple_time = self.get_case_times()
        reftime = datetime.datetime.strptime(self.sim_data.info.attrs['reftime'], ISO_DATETIME_FORMAT)
        fname = xfs.resolve_relative_path(os.path.dirname(self.filename), self.input_files['spec'][0])
        if not os.path.isfile(fname):
            self._logger.warning(f'Unable to find input spectral file {fname}')
            return

        spec_reader = EngReader(fname, idd_to_time, times=simple_time, reftime=reftime)
        spec_cov, num_freqs, min_freqs, delta_freqs = spec_reader.read()
        if spec_cov:
            spec_cov.m_cov.projection = self.grid_proj
            self.query.add_coverage(spec_cov)
            self.query.link_item(self.sim_uuid, spec_cov.m_cov.uuid)
            self.sim_data.info.attrs['spectral_uuid'] = spec_cov.m_cov.uuid
            self.sim_data.info.attrs['num_frequencies'] = num_freqs
            self.sim_data.info.attrs['min_frequency'] = min_freqs
            self.sim_data.info.attrs['delta_frequency'] = delta_freqs

    def read(self):
        """Reads the simulation."""
        self._logger.info('Reading simulation...')
        self.filename = self.read_all_cards()  # read all the sim file cards
        # grid name is the base filename
        project_name = os.path.splitext(os.path.basename(self.filename))[0]

        # build the case data table
        self.get_case_times_for_context()
        if len(self.const_mag) < len(self.case_times):
            self.const_mag.extend([0.0 for _ in range(len(self.const_mag), len(self.case_times))])
        if len(self.const_dir) < len(self.case_times):
            self.const_dir.extend([0.0 for _ in range(len(self.const_dir), len(self.case_times))])
        if len(self.const_surge) < len(self.case_times):
            self.const_surge.extend([0.0 for _ in range(len(self.const_surge), len(self.case_times))])
        time_vals = []
        if len(self.case_times) > 0:
            if isinstance(self.case_times[0], datetime.timedelta):
                time_vals = [case_time.total_seconds() for case_time in self.case_times]
            else:
                time_vals = [(case_time - self._reftime).total_seconds() for case_time in self.case_times]
        time_vals = np.array(time_vals, dtype=np.float64) / 3600.0  # Convert to default units of hours
        time_vals = np.around(time_vals, 3)  # Throw away excess precision
        case_time_data = simulation_data.case_data_table(time_vals, self.const_dir, self.const_mag, self.const_surge)
        self.sim_data.case_times = pandas.DataFrame(case_time_data).to_xarray()

        self.grid_proj.vertical_units = 'METERS'
        self.grid_proj.horizontal_units = 'METERS'
        if 'coord_sys' in self.grid_data:
            self.grid_proj.coordinate_system = self.grid_data['coord_sys']
        if 'spzone' in self.grid_data:
            self.grid_proj.coordinate_zone = self.grid_data['spzone']

        self.build_coverages()  # build the coverages
        self.build_spatial_datasets()
        self.build_grid(project_name)  # build the domain cartesian grid

        if 'spec' in self.input_files:  # build the spectral coverage if there is one
            self.sim_data.info.attrs['boundary_source'] = const.SPEC_OPT_COV
            self.read_spectral_cov()
        else:
            self.sim_data.info.attrs['boundary_source'] = const.OPT_NONE

        self.sim_data.commit()
        self.query.add_simulation(self.do_sim, [self.do_sim_comp])
