"""SimComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.gmi.components2.gmi_sim_component_base import GmiSimComponentBase
from xms.gmi.data.generic_model import Section
from xms.gmi.gui.section_dialog import SectionDialog

# 4. Local modules
from xms.swmm.data.model import get_swmm_model


class SimComponent(GmiSimComponentBase):
    """A hidden Dynamic Model Interface (DMI) component for the SWMM model simulation."""

    def __init__(self, main_file: Optional[str | Path] = None):
        """Initializes the component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        _ = self.data

    def _get_global_parameter_section(self) -> Section:
        """Get the global parameter section."""
        return get_swmm_model().global_parameters

    def _open_model_control(self, dialog_name: str, query: Query, parent: QWidget):
        """
        Run the model control dialog.

        Args:
            dialog_name: A name that can be used as a registry key for loading and saving dialog settings, such as its
                geometry and active selections.
            query: Interprocess communication object.
            parent: Parent widget for the dialog.
        """
        dialog_name = f'{self.module_name}.model_control'
        section = self._get_global_parameter_section()
        values = self.data.global_values
        if values:
            section.restore_values(values)

        dlg = SectionDialog(
            parent=parent,
            section=section,
            dlg_name=dialog_name,
            window_title='Model control',
            enable_unchecked_groups=True,
            hide_checkboxes=True
        )
        if dlg.exec():
            values = dlg.section.extract_values()
            self.data.global_values = values
            self.data.commit()
