"""Miscellaneous stuff we use in tests."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import contextlib
import copy
import os
from random import Random
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.testing import file_comparison
from xms.testing.type_aliases import Pathlike

_random: Random | None = None  # Can be set to create repeatable random uuids (see repeat_uuids())


def new_uuid() -> str:
    """Returns a uuid as a string.

    See repeat_uuids for how to get repeated uuids when testing.
    """
    if _random:
        return str(uuid.UUID(int=_random.getrandbits(128), version=4))
    else:
        return str(uuid.uuid4())


@contextlib.contextmanager
def repeat_uuids():
    """Context manager to make new_uuid() always return the same uuids."""
    global _random
    orig = _random  # Save the original value
    try:
        _random = Random()
        _random.seed(123)
        yield
    finally:
        _random = orig


@contextlib.contextmanager
def env_running_tests(running_tests_str: str):
    """Context manager to change and restore the environment variable XmEnv.ENVIRON_RUNNING_TESTS.

    Args:
        running_tests_str: 'TRUE', 'FALSE', 'MANUAL', or '' for now. '' means don't change anything
    """
    env_var = 'XMS_PYTHON_APP_RUNNING_TESTS'
    orig = os.environ.get(env_var, '')  # Save the original value
    try:
        os.environ[env_var] = running_tests_str  # Change it
        yield
    finally:
        os.environ[env_var] = orig  # Set it back to the original


@contextlib.contextmanager
def env_temp_dir(temp_dir: Pathlike):
    """Context manager to change and restore the environment variable XmEnv.ENVIRON_XMS_TEMP_FOLDER.

    Args:
        temp_dir: If provided, patches 'xms.api.dmi.XmEnv.xms_environ_temp_directory' to return temp_dir.
    """
    env_var = 'XMS_PYTHON_APP_TEMP_DIRECTORY'
    orig = os.environ.get(env_var, '')  # Save the original value
    try:
        if not file_comparison.null_path(temp_dir):
            os.environ[env_var] = str(temp_dir)
        yield
    finally:
        os.environ[env_var] = orig  # Set it back to the original


@contextlib.contextmanager
def env(env_vars: dict[str, str] | None = None):
    """Context manager to temporarily set environment variables for a test.

    Args:
        env_vars: Environment variables to set.
    """
    env_vars = env_vars if env_vars is not None else {}
    orig = copy.deepcopy(os.environ)
    try:
        # Always set the running test flag to 'TRUE', although you could pass something else.
        env_var = 'XMS_PYTHON_APP_RUNNING_TESTS'
        os.environ[env_var] = 'TRUE'

        # Set the environment
        for k, v in env_vars.items():
            os.environ[k] = v
        yield
    finally:
        os.environ.clear()
        for k, v in orig.items():
            os.environ[k] = v
