"""Qt delegate for the dataset selector button column of the user defined constituent table."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, Qt
from PySide2.QtGui import QBrush, QPixmap
from PySide2.QtWidgets import QMessageBox, QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util as tr_util
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg

# 4. Local modules
from xms.tides.gui import gui_util
from xms.tides.gui.tides_pandas_table_model import DSET_TREE_PATH_ROLE


class DsetSelectorButtonDelegate(QStyledItemDelegate):
    """Delegate for the Dset selector button column."""
    def __init__(self, pe_tree, phase_col, parent=None):
        """Initializes the class.

        Args:
            pe_tree (TreeNode): Root of the project tree to display
            phase_col (int): Column index of the phase dataset selectors
            parent (QObject): The parent object.
        """
        super().__init__(parent)
        self._phase_col = phase_col
        self._pe_tree = pe_tree

    def updateEditorGeometry(self, editor, option, index):  # noqa N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)
        # assuming the dset is good since this is just painting the button
        if index.flags() & Qt.ItemIsEnabled:
            dset_path = index.data(DSET_TREE_PATH_ROLE)
            btn = QPushButton()
            if dset_path:
                btn.setText(dset_path)
            else:
                dset_path = gui_util.NULL_SELECTION
                btn.setText(dset_path)
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())

            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)

    def editorEvent(self, event, model, option, index):  # noqa N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                if self._pe_tree and self._pe_tree.children:
                    title = 'Select Tidal Dataset'
                    previous_uuid = index.model().data(index)
                    dlg = TreeItemSelectorDlg(title=title, target_type=xmd.DatasetItem, pe_tree=self._pe_tree,
                                              previous_selection=previous_uuid, parent=self.parent())
                    if dlg.exec():
                        tree_uuid = dlg.get_selected_item_uuid()
                        if tree_uuid:
                            # Using the entire tree path makes the table look ugly. Truncate to tree item name. User
                            # can see full path by pushing the button again.
                            # tree_path = tr_util.build_tree_path(self.pe_tree, tree_uuid)
                            selected_node = tr_util.find_tree_node_by_uuid(self._pe_tree, tree_uuid)
                            index.model().setData(index, tree_uuid)
                            index.model().setData(index, f'.../{selected_node.name}', DSET_TREE_PATH_ROLE)
                        else:
                            index.model().setData(index, '')
                            index.model().setData(index, gui_util.NULL_SELECTION, DSET_TREE_PATH_ROLE)
                else:
                    # Note that this is currently SMS specific. Only supports selecting 2DMesh datasets for now.
                    # This will change in the future.
                    message_box = QMessageBox(self.parent())
                    if index.column() == self._phase_col:
                        msg = 'Create a scalar phase dataset on a 2D mesh, QuadTree, or UGrid to select it for this ' \
                              'constituent.'
                    else:
                        msg = 'Create a scalar amplitude dataset on a 2D mesh, QuadTree, or UGrid to select it for ' \
                              'this constituent.'
                    message_box.setWindowTitle('SMS')
                    message_box.setText(msg)
                    message_box.setIcon(QMessageBox.Warning)
                    message_box.exec_()
        return super().editorEvent(event, model, option, index)
