"""PolygonsFromRasterNodata Algorithm."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from geopandas import GeoDataFrame
from shapely import wkt
from shapely.geometry import Polygon
from shapely.validation import make_valid

# 3. Aquaveo modules
from xms.gdal.rasters import raster_utils as ru
from xms.gdal.utilities import gdal_utils as gu

# 4. Local modules
from xms.tool.utilities.coverage_conversion import convert_polygons_to_coverage, shapely_polygon_list_to_polygons


def polygons_from_raster_nodata(input_raster, cell_threshold, cov_name, default_wkt: str = None,
                                vertical_datum: str = None, vertical_units: str = None) -> GeoDataFrame:
    """
    Grabs the active region of a raster and sets it to a new raster.

    Args:
        input_raster (xms.gdal.rasters.raster_input.RasterInput): The input raster.
        cell_threshold (int): Number of cells required to make a polygon
        cov_name (str): The name of the coverage to create.
        default_wkt (str): The WKT of the default coordinate system
        vertical_datum (str): The vertical datum of the default coordinate system
        vertical_units (str): The vertical units of the default coordinate system

    Returns:
        The new xms Coverage
    """
    vec_ds = ru.convert_raster_data_to_polygons(input_raster)
    vec_lyr = vec_ds.GetLayerByName('bounds')

    # Convert vec_lyr from the raster's projection to the display projection
    coord_trans = None
    display_wkt = None
    if default_wkt is not None:
        display_wkt = gu.add_vertical_to_wkt(default_wkt, vertical_datum, vertical_units, False)
        coord_trans = gu.get_coordinate_transformation(input_raster.wkt, display_wkt)

    # Loop on the polygon features, removing interior rings
    polygon_list = []
    vec_lyr.SetAttributeFilter('VALUE = 1')  # Filter only for polygons with val == 1
    wkt_list = []
    for feature in vec_lyr:
        geom = feature.GetGeometryRef()
        if coord_trans:
            geom.Transform(coord_trans)
        wkt_list.append(geom.ExportToWkt())

    # Determine the size of a cell and then the area required to make a polygon
    pixel_width = abs(input_raster.pixel_width)
    pixel_height = abs(input_raster.pixel_height)
    cell_area = pixel_width * pixel_height
    area_tol = cell_threshold * cell_area

    for cur_wkt in wkt_list:
        # Read in as a shapely polygon, and remove the interior polygons
        polygon_obj = wkt.loads(cur_wkt)
        gtype = polygon_obj.geom_type
        polygon_objs = [polygon_obj] if gtype == 'Polygon' else polygon_obj.geoms if gtype == 'MultiPolygon' else []
        for polygon in polygon_objs:
            polygon_no_holes = Polygon(polygon.exterior.coords, holes=None)
            if polygon_no_holes.area >= area_tol:
                if not polygon_no_holes.is_valid:  # pragma no cover - doesn't get hit after GDAL update
                    # Make the polygon valid and check if we have a multipolygon instead of a polygon
                    valid_polygon = make_valid(polygon_no_holes)
                    if valid_polygon.type == 'Polygon':
                        polygon_no_holes = valid_polygon
                polygon_list.append(polygon_no_holes)
                for interior in polygon_no_holes.interiors:  # pragma no cover - doesn't get hit after GDAL update
                    polygon_interior = Polygon(interior.coords)
                    if polygon_interior.area >= area_tol:
                        polygon_list.append(polygon_interior)
            for interior in polygon.interiors:
                polygon_interior = Polygon(interior.coords)
                if polygon_interior.area >= area_tol:
                    polygon_list.append(polygon_interior)

    polygons = shapely_polygon_list_to_polygons(polygon_list)
    _display_wkt = display_wkt if display_wkt is not None and coord_trans is not None else input_raster.wkt
    new_cov = convert_polygons_to_coverage(polygons, cov_name, _display_wkt)
    vec_ds = None
    return new_cov
