"""Tool to convert a Cartesian grid to a Cartesian UGrid."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['CartesianUGridTool']


# 1. Standard Python modules
from typing import cast
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import GridType
from xms.tool_core import Argument, IoDirection, Tool
from xms.tool_core.grid_argument import GridArgument

# 4. Local modules


ARG_INPUT_GRID = 0
ARG_OUTPUT_NAME = 1


class CartesianUGridTool(Tool):
    """Tool to convert a Cartesian grid to a Cartesian UGrid."""
    def __init__(self, name='Cartesian UGrid from Cartesian Grid'):
        """Initializes the class."""
        super().__init__(name=name)

    def initial_arguments(self) -> list[Argument]:
        """Get initial arguments for tool.

        Must override.

        Returns:
            A list of the initial tool arguments.
        """
        arguments = [
            self.grid_argument(name='input_grid', description='Input grid', io_direction=IoDirection.INPUT),
            self.grid_argument(name='output_grid', description='Name', value='Cartesian UGrid',
                               io_direction=IoDirection.OUTPUT, optional=True),
        ]
        return arguments

    def validate_arguments(self, arguments: list[Argument]) -> dict[str, str]:
        """Validate the tool arguments.

        Args:
            arguments (list): The tool arguments.

        Returns:
            A dictionary of argument errors.
        """
        input_grid = arguments[ARG_INPUT_GRID].text_value
        grid = self.get_input_grid(input_grid)
        if grid.grid_type != GridType.rectilinear_2d:
            return {arguments[ARG_INPUT_GRID].name: 'Must be a rectilinear grid'}
        return {}

    def run(self, arguments: list[Argument]) -> None:
        """Run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        input_grid = arguments[ARG_INPUT_GRID].text_value
        grid = self.get_input_grid(input_grid)
        grid.uuid = str(uuid.uuid4())
        argument = cast(GridArgument, arguments[ARG_OUTPUT_NAME])
        self.set_output_grid(grid, argument, force_ugrid=True)
