"""TrimCoverageTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.algorithms.coverage import TrimCoverage
from xms.tool.utilities.coverage_conversion import get_arcs_from_coverage, get_polygons_from_coverage


class TrimCoverageTool(Tool):
    """Tool to trim arcs in a coverage using a polygon coverage."""
    ARG_INPUT_ARCS_COVERAGE = 0
    ARG_INPUT_POLYGONS_COVERAGE = 1
    ARG_TRIMMING_OPTION = 2
    ARG_BUFFER_DISTANCE = 3
    ARG_OUTPUT_COVERAGE = 4

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Trim Coverage')
        self._num_pts_to_smooth = 2500

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.coverage_argument(name='input_arcs_coverage',
                                   description='Input coverage containing arcs to be trimmed'),
            self.coverage_argument(name='input_polygons_coverage',
                                   description='Input coverage containing polygons to trim by'),
            self.string_argument(name='trimming_option', description='Trimming option', value='Trim to inside',
                                 choices=['Trim to inside', 'Trim to outside']),
            self.float_argument(name='buffer_distance', description='Trimming buffer distance', value=0.0),
            self.coverage_argument(name='output_coverage', description='Output coverage',
                                   io_direction=IoDirection.OUTPUT)
        ]
        return arguments

    def enable_arguments(self, arguments):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments(list): The tool arguments.
        """

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        return errors

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        arcs_coverage = self.get_input_coverage(arguments[self.ARG_INPUT_ARCS_COVERAGE].value)
        arc_data = get_arcs_from_coverage(arcs_coverage)

        polygons_coverage = self.get_input_coverage(arguments[self.ARG_INPUT_POLYGONS_COVERAGE].value)
        hole_data, poly_data = get_polygons_from_coverage(polygons_coverage)
        if hole_data or poly_data:
            trim_out = 1 if arguments[self.ARG_TRIMMING_OPTION].text_value == 'Trim to outside' else 0
            trim_coverage = TrimCoverage(poly_data, hole_data, arc_data, arguments[self.ARG_BUFFER_DISTANCE].value,
                                         trim_out, arguments[self.ARG_OUTPUT_COVERAGE].text_value, self.logger,
                                         self._num_pts_to_smooth)
            output_cov = trim_coverage.generate_coverage()
            self.set_output_coverage(output_cov, arguments[self.ARG_OUTPUT_COVERAGE])
        else:
            self.fail('No polygons defined in the input polygon coverage.')


# def main():
#     """Main function, for testing."""
#     import os
#     from xms.tool_gui.tool_dialog import ToolDialog
#     from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists
#     from xms.tool.utilities.file_utils import get_test_files_path
#
#     ensure_qapplication_exists()
#     tool = TrimCoverageTool()
#     tool.set_gui_data_folder(os.path.join(get_test_files_path(), 'trim_coverage_tool'))
#     arguments = tool.initial_arguments()
#     tool_dialog = ToolDialog(None, tool, arguments, 'Trim Coverage')
#     if tool_dialog.exec():
#         tool.run_tool(tool_dialog.tool_arguments)
#
#
# if __name__ == "__main__":
#     main()
