"""CanopyCoefficientTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import numpy as np
import pandas

# 3. Aquaveo modules
from xms.gdal.utilities import gdal_utils as gu
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.utilities.landuse_codes import get_ccap_canopy_coefficient_values, get_nlcd_canopy_coefficient_values
from xms.tool.utilities.landuse_mapper import LanduseMapper

ARG_INPUT_LANDUSE = 0
ARG_INPUT_INTERP_OPTION = 1
ARG_INPUT_PERCENT_BLOCKED = 2
ARG_INPUT_GRID = 3
ARG_INPUT_LANDUSE_TYPE = 4
ARG_INPUT_CSV = 5
ARG_OUTPUT_DATASET = 6
LOG_FREQUENCY = 10000


class CanopyCoefficientTool(Tool):
    """Tool to a canopy coefficient dataset."""

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Canopy Coefficient')
        self._cogrid = None
        self._grid = None
        self._grid_uuid = None
        self._builder = None
        self._raster = None
        self._raster_values = None
        self._raster_null_value = -999999.0
        self._mapping_table = {}
        self._dataset_vals = None
        self._default_value = 1.0
        self._dataset_values = None

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.raster_argument(name='landuse_raster', description='Input landuse raster'),
            self.string_argument(name='interpolation_option', description='Interpolation option',
                                 choices=['Node location', 'Area around node'], value='Node location'),
            self.float_argument(name='min_percent_blocked',
                                description='Minimum percent blocked which ignores wind stress', value=50.0,
                                min_value=0.0, max_value=100.0),
            self.grid_argument(name='grid', description='Target grid'),
            self.string_argument(name='landuse_type', description='Landuse raster type',
                                 choices=['NLCD', 'C-CAP', 'Other'], value='NLCD'),
            self.file_argument(name='mapping_csv', description='Landuse to canopy coefficient mapping table',
                               optional=True),
            self.dataset_argument(name='canopy_dataset', description='Output canopy coefficient dataset',
                                  value='VCanopy', io_direction=IoDirection.OUTPUT),
        ]
        self.enable_arguments(arguments)
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        # Validate input data
        self._cogrid = self.get_input_grid(arguments[ARG_INPUT_GRID].text_value)
        if not self._cogrid:
            errors[arguments[ARG_INPUT_GRID].name] = 'Could not open target grid.'
        land_use_type = arguments[ARG_INPUT_LANDUSE_TYPE].text_value if \
            arguments[ARG_INPUT_LANDUSE_TYPE].text_value is not None else ''
        csv_file = arguments[ARG_INPUT_CSV].text_value if arguments[ARG_INPUT_CSV].text_value is not None \
            else ''

        # Custom mapping table CSV
        if len(csv_file) > 0:
            if not os.path.exists(csv_file):
                errors[arguments[ARG_INPUT_CSV].name] = 'Could not open Landuse to canopy coefficient mapping table.'
        elif land_use_type == 'Other':  # Must specify a mapping table if using custom land use type
            errors[arguments[ARG_INPUT_CSV].name] = \
                'Must select a canopy coefficient mapping table CSV file when land use type is "Other".'

        return errors

    def enable_arguments(self, arguments):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments(list): The tool arguments.
        """
        custom_codes = arguments[ARG_INPUT_LANDUSE_TYPE].text_value == 'Other'
        arguments[ARG_INPUT_CSV].hide = not custom_codes
        area_option = arguments[ARG_INPUT_INTERP_OPTION].text_value == 'Area around node'
        arguments[ARG_INPUT_PERCENT_BLOCKED].hide = not area_option

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        # Set up some of the grid variables
        self._grid_uuid = self._cogrid.uuid
        self._grid = self._cogrid.ugrid

        # Initialize some of the raster properties, values, etc.
        self._initialize_raster(arguments)

        # Get the canopy coefficient info
        code_vals, _, canopy_coefficient_vals = self.get_canopy_info(arguments)

        if arguments[ARG_INPUT_INTERP_OPTION].text_value == 'Node location':
            point_locations = self._grid.locations.tolist()  # Iterate on a pure Python list, numpy array is slow
            point_locations = gu.transform_points_from_wkt(point_locations, self.default_wkt, self._raster.wkt)
            self._dataset_vals = np.array([self._default_value] * self._grid.point_count)
            code_vals = np.array(code_vals)
            for i, point in enumerate(point_locations):
                column, row = self._raster.coordinate_to_pixel(point[0], point[1])
                code = self._get_raster_value(column, row)
                idx = (np.abs(code_vals - code)).argmin()
                canopy_coefficient = canopy_coefficient_vals[idx]
                self._dataset_vals[i] = 0 if canopy_coefficient else 1
        else:
            percent = arguments[ARG_INPUT_PERCENT_BLOCKED].value * 0.01
            # Calculate the size function and set the dataset values default
            canopy_creator = LanduseMapper(self.logger, self._raster, self._grid, self._default_value,
                                           code_vals, canopy_coefficient_vals, grid_wkt=self.default_wkt)
            self._dataset_vals = canopy_creator.process_points()
            high_mask = self._dataset_vals >= percent
            low_mask = self._dataset_vals < percent
            self._dataset_vals[high_mask] = 0.0
            self._dataset_vals[low_mask] = 1.0

        # Write out the dataset
        self._setup_output_dataset_builder(arguments)
        self._add_output_datasets()

    def _initialize_raster(self, arguments):
        """Get the raster, its size, geo transform, and band, etc."""
        self.logger.info('Retrieving input raster...')
        self._raster = self.get_input_raster(arguments[ARG_INPUT_LANDUSE].text_value)
        self._raster_null_value = self._raster.nodata_value

        # We only need to read the values for the Node location option
        if arguments[ARG_INPUT_INTERP_OPTION].text_value == 'Node location':
            self._raster_values = self._raster.get_raster_values()

    def _setup_output_dataset_builder(self, arguments):
        """Set up dataset builders for selected tool outputs.

        Args:
            arguments (list): The tool arguments.
        """
        # Create a place for the output dataset file
        dataset_name = arguments[ARG_OUTPUT_DATASET].text_value
        self._builder = self.get_output_dataset_writer(
            name=dataset_name,
            geom_uuid=self._grid_uuid,
            null_value=self._raster_null_value,
        )

    def _add_output_datasets(self):
        """Add datasets created by the tool to be sent back to XMS."""
        self.logger.info('Adding output dataset...')
        if self._builder is not None:
            self.logger.info('Writing output canopy coefficient dataset to XMDF file...')
            self._builder.write_xmdf_dataset([0.0], [self._dataset_vals])
            # Send the dataset back to XMS
            self.set_output_dataset(self._builder)

    def _get_raster_value(self, col, row):
        """Get the raster value for a pixel.

        Args:
            col (int): Column of the pixel value
            row (int): Row of the pixel value

        Returns:
            float: The raster value for the pixel or the default canopy coefficient value if out of bounds
        """
        try:
            return self._raster_values[row][col]
        except Exception:
            return self._default_value

    def get_canopy_info(self, arguments):
        """Gets the canopy coefficient information, either built in or from a user CSV file.

        Arguments:
            arguments (list): The tool arguments.

        Returns:
            tuple of lists:  The code values, descriptions, and canopy coefficient values for each landuse type.
        """
        if arguments[ARG_INPUT_LANDUSE_TYPE].text_value == 'NLCD':
            return get_nlcd_canopy_coefficient_values()
        elif arguments[ARG_INPUT_LANDUSE_TYPE].text_value == 'C-CAP':
            return get_ccap_canopy_coefficient_values()
        elif arguments[ARG_INPUT_LANDUSE_TYPE].text_value == 'Other':
            # The user chose a .csv file.  Read it as a dataframe, and rename the columns.
            mapping_df = pandas.read_csv(arguments[ARG_INPUT_CSV].text_value, index_col=0, header=0)
            mapping_df.columns = ['Description', 'Canopy']
            return mapping_df.index.tolist(), mapping_df['Description'].tolist(), mapping_df['Canopy'].tolist()
        return [], [], []
