"""CurvilinearReaderBase class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint.ugrid_builder import UGridBuilder
from xms.datasets.dataset_writer import DatasetWriter
from xms.grid.ugrid import UGrid as XmUGrid

# 4. Local modules


class CurvilinearReaderBase:
    """Base class reader for curvilinear grid files."""

    def __init__(self, grid_name, logger):
        """Initializes the class.

        Args:
            grid_name (str): Optional user input for output grid name. If not
                specified, will try to read from file.
            logger (logging.Logger): The tool logger
        """
        self.logger = logger
        self.grid_name = grid_name
        self._points = []
        self._cellstream = []
        self._cell_elevations = None
        self._i_values = []  # Cell-based i-coordinate dataset, in cellstream order
        self._j_values = []  # Cell-based j-coordinate dataset, in cellstream order
        self._cogrid_uuid = str(uuid.uuid4())

    def create_cogrid(self):
        """Creates the CoGrid object to send back to XMS.

        Returns:
            tuple(CoGrid, wkt): The output curvilinear CoGrid and its WKT for projection
        """
        self.logger.info('Building UGrid...')
        xmugrid = XmUGrid(self._points, self._cellstream)
        co_builder = UGridBuilder()
        co_builder.set_unconstrained()
        co_builder.set_ugrid(xmugrid)
        co_grid = co_builder.build_grid()
        co_grid.uuid = self._cogrid_uuid

        # Set the cell elevations if we have them
        if self._cell_elevations is not None:
            co_grid.cell_elevations = self._cell_elevations

        # For now, we are always going to assume meter units
        wkt = 'LOCAL_CS["None",LOCAL_DATUM["None",32767],UNIT["Meters",1.0],AXIS["None",OTHER]]'
        return co_grid, wkt

    def create_datasets(self):
        """Create cell-based datasets for the output grid.

        Returns:
            list[DatasetWriter]: The output datasets
        """
        self.logger.info('Writing output cell I-coordinate dataset...')
        dset_uuid = str(uuid.uuid4())
        i_writer = DatasetWriter(name='Cell I', dset_uuid=dset_uuid,
                                 geom_uuid=self._cogrid_uuid, location='cells')
        i_writer.write_xmdf_dataset([0.0], [self._i_values])
        self.logger.info('Writing output cell J-coordinate dataset...')
        dset_uuid = str(uuid.uuid4())
        j_writer = DatasetWriter(name='Cell J', dset_uuid=dset_uuid,
                                 geom_uuid=self._cogrid_uuid, location='cells')
        j_writer.write_xmdf_dataset([0.0], [self._j_values])
        return [i_writer, j_writer]
