"""FillNodataTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gdal.rasters import raster_utils as ru
from xms.gdal.rasters import RasterInput
from xms.tool_core import IoDirection, Tool

# 4. Local modules

ARG_INPUT_RASTER = 0
ARG_INPUT_MAX_DISTANCE = 1
ARG_INPUT_SMOOTH_ITERATIONS = 2
ARG_OUTPUT_RASTER = 3


class FillNodataTool(Tool):
    """Tool to fill nodata in a raster, creating a new raster."""

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Fill Nodata')

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.raster_argument(name='input_raster', description='Input raster'),
            self.integer_argument(name='max_distance', description='Maximum distance (in pixels) to interpolate',
                                  min_value=1, value=100),
            self.integer_argument(name='smooth_iterations',
                                  description='The number of 3x3 average filter smoothing iterations to run',
                                  min_value=0, value=0, optional=True),
            self.raster_argument(name='output_raster', description='Output raster', io_direction=IoDirection.OUTPUT)
        ]
        return arguments

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        input_raster = self.get_input_raster(arguments[ARG_INPUT_RASTER].value)
        max_distance = int(arguments[ARG_INPUT_MAX_DISTANCE].value)
        smooth_iterations = 0 if not arguments[ARG_INPUT_SMOOTH_ITERATIONS].value else \
            int(arguments[ARG_INPUT_SMOOTH_ITERATIONS].value)
        out_path = self.get_output_raster(arguments[ARG_OUTPUT_RASTER].value)

        if self._fill_nodata(input_raster, max_distance, smooth_iterations, out_path):
            out_path = ru.reproject_raster(out_path, self.get_output_raster(arguments[ARG_OUTPUT_RASTER].value),
                                           self.default_wkt, self.vertical_datum, self.vertical_units)
            self.set_output_raster_file(out_path, arguments[ARG_OUTPUT_RASTER].text_value)

    def _fill_nodata(self, input_raster, max_distance, smooth_iterations, out_path):
        """
        Fills nodata values into a new raster.

        Args:
            input_raster (RasterInput): The input raster.
            max_distance (int): The maximum distance in pixels to search out to interpolate.
            smooth_iterations (int): The number of 3x3 average filter smoothing iterations to run.
            out_path (str): The output path of the new raster.
        """
        ru.copy_raster_from_raster_input(input_raster, out_path)
        ru.fill_nodata(RasterInput(out_path, True), max_distance, smooth_iterations)
        return True
