"""ExportProjectTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 3. Aquaveo modules
from xms.grid.ugrid import UGrid
from xms.tool_core import Argument, IoDirection, Tool

# 2. Third party modules

# 4. Local modules
from xms.tool.algorithms.schism import write_horizontal_grid_file


class ExportHorizontalGridTool(Tool):
    """Tool to export a SCHISM horizontal grid file."""
    ARG_INPUT_GRID = 0
    ARG_OUTPUT_FILE = 1

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Export Horizontal Grid')

    def initial_arguments(self) -> list[Argument]:
        """Get initial arguments for tool.

        Must override.

        Returns:
            A list of the initial tool arguments.
        """
        arguments = [
            self.grid_argument(name='input_grid', description='Input grid'),
            self.file_argument(name='output_file', description='Output file', file_filter='Horizontal Grid (*.gr3)',
                               default_suffix='gr3', io_direction=IoDirection.OUTPUT),
        ]
        return arguments

    def validate_arguments(self, arguments: list[Argument]) -> dict[str, str]:
        """Validate the tool arguments.

        Args:
            arguments (list): The tool arguments.
        """
        input_grid = arguments[self.ARG_INPUT_GRID].text_value
        grid = self.get_input_grid(input_grid)
        if not grid.check_all_cells_are_of_types([UGrid.cell_type_enum.QUAD, UGrid.cell_type_enum.TRIANGLE]):
            return {arguments[self.ARG_INPUT_GRID].name: 'Must contain only quads or triangles.'}
        return {}

    def run(self, arguments: list[Argument]) -> None:
        """Run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        grid = self.get_input_grid(arguments[self.ARG_INPUT_GRID].text_value)
        output_file = Path(arguments[self.ARG_OUTPUT_FILE].text_value)
        with open(output_file, 'w') as file:
            write_horizontal_grid_file(file, grid.ugrid, output_file.name, self.logger)
