"""ImportCurvilinearGridTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.file_io.curvilinear.ch3d_grid_reader import Ch3dGridReader
from xms.tool.file_io.curvilinear.efdc_grid_reader import EfdcGridReader

ARG_INPUT_FORMAT = 0
ARG_INPUT_CH3D_FILE = 1
ARG_INPUT_DXDY_FILE = 2
ARG_INPUT_LXLY_FILE = 3
ARG_OUTPUT_GRID = 4
ARG_OUTPUT_I_DSET = 5
ARG_OUTPUT_J_DSET = 6


class ImportCurvilinearGridTool(Tool):
    """Tool to import a curvilinear grid from a CH3D or EFDC file."""

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Import Curvilinear Grid')

    def _add_outputs(self, reader, arguments):
        """Creates the CoGrid object to send back to XMS.

        Args:
            reader (CurvilinearReaderBase): The curvilinear grid file reader
            arguments (list): The tool arguments.
        """
        self._add_output_cogrid(reader, arguments)
        self._add_output_datasets(reader)

    def _add_output_cogrid(self, reader, arguments):
        """Creates the CoGrid object to send back to XMS.

        Args:
            reader (CurvilinearReaderBase): The curvilinear grid file reader
            arguments (list): The tool arguments.
        """
        # The output grid name is an optional user input and may have been read from the file.
        grid_name = reader.grid_name if reader.grid_name else f'{arguments[ARG_INPUT_FORMAT].value} Curvilinear Grid'
        arguments[ARG_OUTPUT_GRID].value = grid_name
        # Build the curvilinear CoGrid from imported data.
        co_grid, wkt = reader.create_cogrid()
        self.set_output_grid(co_grid, arguments[ARG_OUTPUT_GRID], projection=wkt)
        self.logger.info('Curvilinear UGrid successfully created.')

    def _add_output_datasets(self, reader):
        """Creates the output datasets to send back to SMS.

        Args:
            reader (CurvilinearReaderBase): The curvilinear grid file reader
        """
        self.logger.info('Building output datasets...')
        # Add cell-based i and j coordinate datasets
        dsets = reader.create_datasets()
        for dset in dsets:
            self.set_output_dataset(dset)
        self.logger.info('Output datasets successfully created.')

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        if arguments[ARG_INPUT_FORMAT].value == 'CH3D':
            if not os.path.isfile(arguments[ARG_INPUT_CH3D_FILE].value):
                errors[arguments[ARG_INPUT_CH3D_FILE].name] = 'Could not find file.'
        elif arguments[ARG_INPUT_FORMAT].value == 'EFDC':
            if not os.path.isfile(arguments[ARG_INPUT_DXDY_FILE].value):
                errors[arguments[ARG_INPUT_DXDY_FILE].name] = 'Could not find file.'
            if not os.path.isfile(arguments[ARG_INPUT_LXLY_FILE].value):
                errors[arguments[ARG_INPUT_LXLY_FILE].name] = 'Could not find file.'
        return errors

    def enable_arguments(self, arguments):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments(list): The tool arguments.
        """
        is_ch3d = arguments[ARG_INPUT_FORMAT].value == 'CH3D'
        arguments[ARG_INPUT_CH3D_FILE].hide = not is_ch3d
        arguments[ARG_INPUT_DXDY_FILE].hide = is_ch3d
        arguments[ARG_INPUT_LXLY_FILE].hide = is_ch3d

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.string_argument(name='format', description='Format of the data file', value='CH3D',
                                 choices=['CH3D', 'EFDC']),
            self.file_argument(name='ch3d_file', description='CH3D formatted grid file', optional=True),
            self.file_argument(name='dxdy_file', description='EFDC formatted dxdy.inp file',
                               optional=True),
            self.file_argument(name='lxly_file', description='EFDC formatted lxly.inp file',
                               optional=True),
            self.grid_argument(name='output_ugrid', description='Output UGrid name', optional=True,
                               io_direction=IoDirection.OUTPUT),
            self.dataset_argument(name='i_dset', description='Output cell i-coordinate dataset', hide=True, value='I',
                                  io_direction=IoDirection.OUTPUT),
            self.dataset_argument(name='j_dset', description='Output cell j-coordinate dataset', hide=True, value='J',
                                  io_direction=IoDirection.OUTPUT),
        ]
        self.enable_arguments(arguments)
        return arguments

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        if arguments[ARG_INPUT_FORMAT].value == 'CH3D':
            self.logger.info('Reading curvilinear grid from CH3D formatted file...')
            reader = Ch3dGridReader(arguments[ARG_INPUT_CH3D_FILE].value, arguments[ARG_OUTPUT_GRID].value,
                                    self.logger)
        else:  # arguments[ARG_INPUT_FORMAT].value == 'EFDC':
            self.logger.info('Reading curvilinear grid from EFDC formatted file...')
            reader = EfdcGridReader(arguments[ARG_INPUT_DXDY_FILE].value, arguments[ARG_INPUT_LXLY_FILE].value,
                                    arguments[ARG_OUTPUT_GRID].value, self.logger)
        reader.read()
        self._add_outputs(reader, arguments)
