"""Landuse code utilities."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


def get_nlcd_codes_and_descriptions():
    """Gets the common NLCD code values and descriptions.

    Returns:
        (tuple):  Tuple of list of default NLCD code values, and list of NLCD descriptions.
    """
    code_vals = [0,
                 1,  # Doesn't really exist, but keeps arrays consistent with C-CAP
                 11,
                 12,
                 21,
                 22,
                 23,
                 24,
                 31,
                 41,
                 42,
                 43,
                 51,
                 52,
                 71,
                 72,
                 73,
                 74,
                 81,
                 82,
                 90,
                 95,
                 91,
                 92,
                 93,
                 94,
                 96,
                 97,
                 98,
                 99]
    descriptions = ['Background',
                    'Unclassified',
                    'Open Water',
                    'Perennial Ice/Snow',
                    'Developed, Open Space',
                    'Developed, Low Intensity',
                    'Developed, Medium Intensity',
                    'Developed, High Intensity',
                    'Barren Land (Rock/Sand/Clay)',
                    'Deciduous Forest',
                    'Evergreen Forest',
                    'Mixed Forest',
                    'Dwarf Scrub',
                    'Shrub/Scrub',
                    'Grassland/Herbaceous',
                    'Sedge/Herbaceous',
                    'Lichens',
                    'Moss',
                    'Pasture/Hay',
                    'Cultivated Crops',
                    'Woody Wetlands',
                    'Emergent Herbaceous Wetlands',
                    'Palustrine Forested Wetland',
                    'Palustrine Scrub/Shrub Wetland',
                    'Estuarine Forested Wetland',
                    'Estuarine Scrub/Shrub Wetland',
                    'Palustrine Emergent Wetland (Persistant)',
                    'Estuarine Emergent Wetland',
                    'Palustrine Aquatic Bed',
                    'Estuarine Aquatic Bed']
    return code_vals, descriptions


def get_ccap_codes_and_descriptions():
    """Gets the common C-CAP code values and descriptions.

    Returns:
        (tuple):  Tuple of list of default C-CAP code values, and list of C-CAP descriptions.
    """
    code_vals = [0,
                 1,
                 2,
                 3,
                 4,
                 5,
                 6,
                 7,
                 8,
                 9,
                 10,
                 11,
                 12,
                 13,
                 14,
                 15,
                 16,
                 17,
                 18,
                 19,
                 20,
                 21,
                 22,
                 23,
                 24,
                 25]
    descriptions = ["Background",
                    "Unclassified",
                    "Developed, High Intensity",
                    "Developed, Medium Intensity",
                    "Developed, Low Intensity",
                    "Developed, Open Space",
                    "Cultivated Crops",
                    "Pasture/Hay",
                    "Grassland/Herbaceous",
                    "Deciduous Forest",
                    "Evergreen Forest",
                    "Mixed Forest",
                    "Scrub/Shrub",
                    "Palustrine Forested Wetland",
                    "Palustrine Scrub/Shrub Wetland",
                    "Palustrine Emergent Wetland (Persistent)",
                    "Estuarine Forested Wetland",
                    "Estuarine Scrub/Shrub Wetland",
                    "Estuarine Emergent Wetland",
                    "Unconsolidated Shore",
                    "Barren Land",
                    "Open Water",
                    "Palustrine Aquatic Bed",
                    "Estuarine Aquatic Bed",
                    "Perennial Ice/Snow",
                    "Tundra"]
    return code_vals, descriptions


def get_default_mannings_n_values():
    """Gets code values for default Mannings N values.

    Returns:
        (tuple):  Tuple of list of default Mannings N codes, list of descriptions, and list of N values
    """
    code_vals, descriptions = get_nlcd_codes_and_descriptions()
    mannings_val = [0.025,
                    0.025,
                    0.02,
                    0.01,
                    0.02,
                    0.05,
                    0.1,
                    0.15,
                    0.09,
                    0.1,
                    0.11,
                    0.1,
                    0.04,
                    0.05,
                    0.034,
                    0.03,
                    0.027,
                    0.025,
                    0.033,
                    0.037,
                    0.1,
                    0.045,
                    0.1,
                    0.048,
                    0.1,
                    0.048,
                    0.045,
                    0.045,
                    0.015,
                    0.015]
    return code_vals, descriptions, mannings_val


def get_ccap_mannings_n_values():
    """Gets code values for default Mannings N values, for NOAA C-CAP.

    See:  https://coast.noaa.gov/data/digitalcoast/pdf/ccap-class-scheme-regional.pdf
    Note that the data we use is C-CAP data from 2010... Tundra and Perennial Ice/Snow are flipped for that.
    See:  https://coast.noaa.gov/arcgis/rest/services/CCAP/conus_10/MapServer/legend

    Returns:
        (tuple):  Tuple of list of default Mannings N codes, list of descriptions, and list of N values
    """
    code_vals, descriptions = get_ccap_codes_and_descriptions()
    mannings_val = [0.025,
                    0.025,
                    0.15,
                    0.1,
                    0.05,
                    0.02,
                    0.037,
                    0.033,
                    0.034,
                    0.1,
                    0.11,
                    0.1,
                    0.05,
                    0.1,
                    0.048,
                    0.045,
                    0.1,
                    0.048,
                    0.045,
                    0.03,
                    0.09,
                    0.02,
                    0.015,
                    0.015,
                    0.01,
                    0.03]
    return code_vals, descriptions, mannings_val


def get_directional_roughness_values():
    """Gets code values for directional roughness values.

    Returns:
        (tuple):  Tuple of list of directional roughness codes, list of descriptions, and list of roughness values
    """
    code_vals, descriptions = get_nlcd_codes_and_descriptions()
    roughness_vals = [0.0,
                      0.0,
                      0.001,
                      0.012,
                      0.1,
                      0.3,
                      0.4,
                      0.55,
                      0.04,
                      0.65,
                      0.72,
                      0.71,
                      0.1,
                      0.12,
                      0.04,
                      0.03,
                      0.025,
                      0.02,
                      0.06,
                      0.06,
                      0.55,
                      0.11,
                      0.55,
                      0.12,
                      0.55,
                      0.12,
                      0.11,
                      0.11,
                      0.03,
                      0.03]
    return code_vals, descriptions, roughness_vals


def get_directional_roughness_values_ccap():
    """Gets code values for directional roughness values, for C-CAP.

    Returns:
        (tuple):  Tuple of list of directional roughness codes, list of descriptions, and list of roughness values
    """
    code_vals, descriptions = get_ccap_codes_and_descriptions()
    roughness_vals = [0.0,
                      0.0,
                      0.55,
                      0.4,
                      0.3,
                      0.1,
                      0.06,
                      0.06,
                      0.04,
                      0.65,
                      0.72,
                      0.71,
                      0.12,
                      0.55,
                      0.12,
                      0.11,
                      0.55,
                      0.12,
                      0.11,
                      0.03,
                      0.04,
                      0.001,
                      0.03,
                      0.03,
                      0.012,
                      0.03]
    return code_vals, descriptions, roughness_vals


def get_nlcd_canopy_coefficient_values():
    """Gets canopy coefficient values, for NLCD.

    Returns:
        (tuple):  Tuple of list of directional roughness codes, list of descriptions, and list of roughness values
    """
    code_vals, descriptions = get_nlcd_codes_and_descriptions()
    roughness_vals = [0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      1,  # Deciduous Forest
                      1,  # Evergreen Forest
                      1,  # Mixed Forest
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      1,  # Woody Wetlands
                      0,
                      1,  # Palustrine Forested Wetland
                      1,  # Palustrine Scrub/Shrub Wetland
                      1,  # Estuarine Forested Wetland
                      0,
                      0,
                      0,
                      0,
                      0]
    return code_vals, descriptions, roughness_vals


def get_ccap_canopy_coefficient_values():
    """Gets canopy coefficient values, for C-CAP.

    Returns:
        (tuple):  Tuple of list of directional roughness codes, list of descriptions, and list of roughness values
    """
    code_vals, descriptions = get_ccap_codes_and_descriptions()
    roughness_vals = [0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      1,
                      1,
                      1,
                      0,
                      1,
                      1,
                      0,
                      1,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0,
                      0]
    return code_vals, descriptions, roughness_vals


def get_nlcd_quadratic_friction_values():
    """Gets quadratic friction values, for NLCD.

    Returns:
        (tuple):  Tuple of list of NLCD codes, list of descriptions, and list of quadratic friction values.
    """
    code_vals, descriptions = get_nlcd_codes_and_descriptions()
    roughness_vals = [0.002,
                      0.002,
                      0.0018,
                      0.000460,
                      0.0018,
                      0.011,
                      0.046,
                      0.1,
                      0.037,
                      0.046,
                      0.055,
                      0.046,
                      0.0073,
                      0.011,
                      0.0053,
                      0.0041,
                      0.0033,
                      0.0028,
                      0.005,
                      0.0062,
                      0.046,
                      0.0092,
                      0.046,
                      0.01,
                      0.046,
                      0.01,
                      0.0092,
                      0.0092,
                      0.001,
                      0.001]
    return code_vals, descriptions, roughness_vals


def get_ccap_quadratic_friction_values():
    """Gets quadratic friction values, for C-CAP.

    Returns:
        (tuple):  Tuple of list of C-CAP landuse codes, list of descriptions, and list of quadratic friction values.
    """
    code_vals, descriptions = get_ccap_codes_and_descriptions()
    friction_vals = [0.002,
                     0.002,
                     0.1,
                     0.046,
                     0.011,
                     0.0018,
                     0.0062,
                     0.005,
                     0.0053,
                     0.046,
                     0.055,
                     0.046,
                     0.011,
                     0.046,
                     0.01,
                     0.0092,
                     0.046,
                     0.01,
                     0.0092,
                     0.002,  # Unconsolidated Shore
                     0.037,
                     0.0018,
                     0.001,
                     0.001,
                     0.00046,
                     0.002]  # Tundra
    return code_vals, descriptions, friction_vals


def get_nlcd_chezy_friction_values():
    """Gets Chezy friction values, for NLCD.

    Returns:
        (tuple):  Tuple of list of NLCD codes, list of descriptions, and list of Chezy friction values.
    """
    code_vals, descriptions = get_nlcd_codes_and_descriptions()
    chezy_vals = [60.0,
                  60.0,
                  110.0,
                  220.0,
                  110.0,
                  44.0,
                  22.0,
                  15.0,
                  24.0,
                  22.0,
                  20.0,
                  22.0,
                  55.0,
                  44.0,
                  64.0,
                  73.0,
                  81.0,
                  87.0,
                  66.0,
                  59.0,
                  22.0,
                  48.0,
                  22.0,
                  45.0,
                  22.0,
                  45.0,
                  48.0,
                  48.0,
                  150.0,
                  150.0]
    return code_vals, descriptions, chezy_vals


def get_ccap_chezy_friction_values():
    """Gets Chezy friction values, for C-CAP.

    Returns:
        (tuple):  Tuple of list of C-CAP landuse codes, list of descriptions, and list of Chezy friction values.
    """
    code_vals, descriptions = get_ccap_codes_and_descriptions()
    chezy_vals = [60.0,
                  60.0,
                  15.0,
                  22.0,
                  44.0,
                  110.0,
                  59.0,
                  66.0,
                  64.0,
                  22.0,
                  20.0,
                  22.0,
                  44.0,
                  22.0,
                  45.0,
                  48.0,
                  22.0,
                  45.0,
                  48.0,
                  60.0,  # Unconsolidated Shore
                  24.0,
                  110.0,
                  150.0,
                  150.0,
                  220.0,
                  60.0]  # Tundra
    return code_vals, descriptions, chezy_vals
