"""WhiteboxTool base class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import json

# 2. Third party modules


# 3. Aquaveo modules
from xms.core.filesystem import filesystem
from xms.gdal.rasters import raster_utils as ru
from xms.gdal.rasters import RasterInput
from xms.gdal.utilities import gdal_utils as gu
from xms.gdal.utilities import GdalRunner
from xms.gdal.vectors import VectorInput
from xms.tool_core import IoDirection, Tool
from xms.wbtools.whitebox_tools import WhiteboxTools

# 4. Local modules
from xms.tool.utilities.coverage_conversion import (arcs_to_shapefile, convert_lines_to_coverage,
                                                    convert_points_to_coverage, convert_polygons_to_coverage,
                                                    get_polygons_from_coverage, points_to_shapefile,
                                                    polygons_to_shapefile)
from xms.tool.utilities.file_utils import get_csv_filename, get_raster_filename, get_vector_filename


def fix_name_and_add_spaces(tool_name):
    """Adds spaces before the capital letters in the CamelCase tool_name, replaces 'Pointer' with 'Flow Directions'."""
    new_string = ""
    for c in tool_name:
        if c.isupper():
            new_string += " " + c
        else:
            new_string += c
    # Replace "Pointer" with "Flow Directions".  We don't use the "Pointer" terminology.
    if 'Pointer' in new_string:
        new_string = new_string.replace('Pointer', 'Flow Directions')
    return new_string.strip()


def get_python_friendly_name(parameter_name):
    """Converts parameter_name to a python friendly name."""
    return parameter_name.strip('*?., ').lower().replace(' ', '_').replace('-', '_').replace('(', '').replace(')', '')


def set_output_data(raster_wkt, arguments, tool, info, ignored_optional_args, ignored_arg_values, send_outputs_to_xms):
    """Sets the output data for the tool.

    Args:
        raster_wkt (str): The WKT from the raster file.
        arguments (list): A list of the tool's arguments.
        tool (xms.tool_core.Tool): The tool.
        info (dict): The WBT tool information.
        ignored_optional_args (list): A list of the python-friendly names of optional arguments for which the default
         value will be used.
        ignored_arg_values (dict): A dict containing python-friendly argument names as the key and their values
         for which the values will be given to the whitebox tool.
        send_outputs_to_xms (bool): Whether the outputs (raster and/or vector data) from running the tool will be
         sent to XMS after the tool is run.
    """
    arg_index = 0
    for p in info['parameters']:
        name = get_python_friendly_name(p['name'])
        if name in ignored_optional_args:
            continue
        p_type = p['parameter_type']
        if 'NewFile' in p_type:
            file_type = p_type['NewFile']
            if 'Raster' in file_type:
                file = None
                if name in ignored_arg_values:
                    file = ignored_arg_values[name]
                elif arguments:
                    file = arguments[arg_index].value
                if file is not None and send_outputs_to_xms:
                    raster_file = get_raster_filename(file)
                    ru.reproject_raster(raster_file, raster_file, tool.default_wkt, tool.vertical_datum,
                                        tool.vertical_units)
                    tool.set_output_raster_file(tool.get_output_raster(file), file)
            elif 'Vector' in file_type:
                file = None
                if name in ignored_arg_values:
                    file = ignored_arg_values[name]
                elif arguments:
                    file = arguments[arg_index].value
                if file is not None:
                    vector_filename = get_vector_filename(file)
                    vi = VectorInput(vector_filename)
                    layer_type = vi.layer_type
                    new_cov = None
                    cov_geometry = None
                    display_wkt = None
                    if tool.default_wkt is not None:
                        display_wkt = gu.add_vertical_to_wkt(tool.default_wkt, tool.vertical_datum,
                                                             tool.vertical_units)
                    match layer_type:
                        case VectorInput.wkbLineString | VectorInput.wkbLineStringM | VectorInput.wkbLineStringZM \
                                | VectorInput.wkbLineString25D:
                            cov_geometry = vi.get_line_features()
                            new_cov = convert_lines_to_coverage(cov_geometry, file, raster_wkt, display_wkt)
                        case VectorInput.wkbPoint | VectorInput.wkbPointM | VectorInput.wkbPointZM \
                                | VectorInput.wkbPoint25D:
                            cov_geometry = vi.get_point_features()
                            new_cov = convert_points_to_coverage(cov_geometry, file, raster_wkt, display_wkt)
                        case VectorInput.wkbPolygon | VectorInput.wkbPolygonM | VectorInput.wkbPolygonZM \
                                | VectorInput.wkbPolygon25D:
                            cov_geometry = vi.get_poly_features()
                            new_cov = convert_polygons_to_coverage(cov_geometry, file, raster_wkt, display_wkt)
                    if not new_cov.empty:
                        if send_outputs_to_xms:
                            if name in ignored_arg_values:
                                argument = tool.coverage_argument(name=name, description=name,
                                                                  io_direction=IoDirection.OUTPUT,
                                                                  value=ignored_arg_values[name])
                            else:
                                argument = arguments[arg_index]
                            tool.set_output_coverage(new_cov, argument)
                        try:
                            # Add this coverage to the list of vector outputs to use it later.
                            tool.vector_output.append(new_cov)
                        except AttributeError:
                            pass
                    elif cov_geometry is not None:  # pragma no cover - for linux tests
                        try:
                            # Add the coverage geometry to the list of vector outputs to use it later.
                            tool.vector_output.append(cov_geometry)
                        except AttributeError:
                            pass
        if name not in ignored_arg_values and arguments:
            arg_index += 1


def parse_arguments(arguments, tool, info, ignored_optional_args, ignored_arg_values):
    """Sets the output data for the tool.

    Args:
        arguments (list): A list of the tool's arguments.
        tool (xms.tool_core.Tool): The tool.
        info (dict): The WBT tool information.
        ignored_optional_args (list): A list of the python-friendly names of optional arguments for which the default
         value will be used.
        ignored_arg_values (dict): A dict containing python-friendly argument names as the key and their values
         for which the values will be given to the whitebox tool.
    """
    raster_wkt = ''
    display_wkt = tool.default_wkt
    wbt_args = []
    # Process input rasters first to get the raser_wkt.  Then process everything else.
    arg_index = 0
    for p in info['parameters']:
        name = get_python_friendly_name(p['name'])
        if name in ignored_optional_args:
            continue
        p_type = p['parameter_type']
        if 'ExistingFile' in p_type:
            file_type = p_type['ExistingFile']
            if 'Raster' in file_type:
                file = None
                if name in ignored_arg_values:
                    file = ignored_arg_values[name]
                elif arguments:
                    file = arguments[arg_index].value
                if file is not None:
                    raster_wkt, filename = _get_filename(p_type, tool, file, raster_wkt, display_wkt)
                    wbt_args.append(f"{p['flags'][-1]}='{filename}'")
        if name not in ignored_arg_values and arguments:
            arg_index += 1
    # Now process all the other arguments.
    arg_index = 0
    for p in info['parameters']:
        name = get_python_friendly_name(p['name'])
        if name in ignored_optional_args:
            continue
        p_type = p['parameter_type']
        if 'ExistingFile' in p_type or 'NewFile' in p_type:
            add_argument = True
            if 'ExistingFile' in p_type:
                file_type = p_type['ExistingFile']
                if 'Raster' in file_type:
                    add_argument = False
            if add_argument:
                file = None
                if name in ignored_arg_values:
                    file = ignored_arg_values[name]
                elif arguments:
                    file = arguments[arg_index].value
                if file is not None:
                    raster_wkt, filename = _get_filename(p_type, tool, file, raster_wkt, display_wkt)
                    wbt_args.append(f"{p['flags'][-1]}='{filename}'")
        elif 'Boolean' in p_type:
            value = False
            if name in ignored_arg_values:
                value = ignored_arg_values[name]
            elif arguments:
                value = arguments[arg_index].value
            if value:
                wbt_args.append(f"{p['flags'][-1]}")
        elif 'ExistingFileOrFloat' in p_type or 'Float' in p_type or 'Integer' in p_type or 'Text' in p_type or \
                'String' in p_type or 'StringOrNumber' in p_type or 'VectorAttributeField' in p_type:
            value = None
            if name in ignored_arg_values:
                value = ignored_arg_values[name]
            elif arguments:
                value = arguments[arg_index].value
            if value is not None:
                wbt_args.append(f"{p['flags'][-1]}={value}")
        elif 'OptionList' in p_type:
            value = None
            if name in ignored_arg_values:
                value = ignored_arg_values[name]
            elif arguments:
                value = arguments[arg_index].value
            if value is not None:
                wbt_args.append(f"{p['flags'][-1]}='{value}'")
        else:
            tool.logger.warning(f"Unsupported parameter type: {p_type}.")
        if name not in ignored_arg_values and arguments:
            arg_index += 1
    return raster_wkt, wbt_args


def _get_filename(p_type, tool, file, raster_wkt, display_wkt):
    file_type = ''
    io_direction = IoDirection.INPUT
    if 'ExistingFile' in p_type:
        file_type = p_type['ExistingFile']
    elif 'NewFile' in p_type:
        file_type = p_type['NewFile']
        io_direction = IoDirection.OUTPUT
    if 'Raster' in file_type:
        if io_direction == IoDirection.INPUT:
            raster_input_file = None
            try:
                raster_input_file = tool.get_input_raster_file(file)
            except KeyError:
                pass
            if raster_input_file is None:
                raster_input_file = get_raster_filename(file)
            raster_format = ru.get_raster_format_short_name(raster_input_file)
            if raster_format not in {'GTiff', 'AAIGrid', 'GRASSASCIIGrid', 'GSAG', 'UNDEFINED'}:
                # Convert to a geotiff.
                gdal_runner = GdalRunner()
                args = ['-of', 'GTiff']
                raster_input_file = gdal_runner.run_wrapper('gdal_translate', raster_input_file, 'geotiff_file.tif',
                                                            args)
            input_raster = RasterInput(raster_input_file)
            if input_raster:
                raster_wkt = input_raster.wkt
            return raster_wkt, raster_input_file
        else:
            return raster_wkt, get_raster_filename(file)
    elif 'Vector' in file_type or 'RasterAndVector' in file_type:
        coverage_defined = False
        if io_direction == IoDirection.INPUT:
            coverage = tool.get_input_coverage(file)
            if coverage is not None:
                points = coverage[coverage['geometry_types'] == 'Point']
                arcs = coverage[coverage['geometry_types'] == 'Arc']
                polygons = coverage[coverage['geometry_types'] == 'Polygon']
                file = filesystem.temp_filename(suffix='.shp')
                if not polygons.empty:
                    polygons_to_shapefile(get_polygons_from_coverage(polygons), file, raster_wkt, display_wkt, True)
                elif not arcs.empty:
                    arcs_to_shapefile(arcs, file, raster_wkt, display_wkt, True)
                elif not points.empty:
                    points_to_shapefile(points, file, raster_wkt, display_wkt, True)
                coverage_defined = True
        if not coverage_defined:
            if file is not None:
                vector_file = get_vector_filename(file)
                return raster_wkt, vector_file
    elif 'Csv' in file_type:
        return raster_wkt, get_csv_filename(file)
    if file is not None:
        return raster_wkt, file


def get_argument_from_name(arguments, name):
    """Gets the tool argument from the given name.

    Args:
        arguments (list): A list of the tool's arguments.
        name (str): The argument name.

    Returns:
        (tool.Argument): The tool's argument or None if the argument does not exist
    """
    for argument in arguments:
        if argument.name == name:
            return argument
    return None


class WhiteboxTool(Tool):
    """WhiteboxTool base class."""

    def __init__(self, wbt_name, tool_name=None, ignored_optional_args=None, ignored_arg_values=None,
                 arg_descriptions=None):
        """Initializes the class.

        Args:
            wbt_name (str): The whitebox tool class name. This is also used as the tool name is no tool_name is defined.
            tool_name (str): The tool name, if desired. If this is None, the wbt_name is converted to the tool name.
             This must match the tool name in tools.xml if the converted wbt_name does not match the tool name.
            ignored_optional_args (list): A list of the python-friendly names of optional arguments that will not be
             displayed in the tool dialog and for which the default value will be used.
            ignored_arg_values (dict): A dict containing python-friendly argument names as the key and their values
             that will not be displayed in the tool dialog but for which the values will be given to the whitebox tool.
            arg_descriptions (dict): A dict containing python-friendly argument names as the key and the argument
             description(s) to use in the GUI for the values.
        """
        super().__init__(name=fix_name_and_add_spaces(wbt_name) if tool_name is None else tool_name)
        self.wbt_name = wbt_name
        self.wbt = WhiteboxTools()
        self.wbt.set_compress_rasters(False)
        self.wbt.work_dir = ''
        self.wbt.set_default_callback(self.logger.info)
        if ignored_optional_args is None:
            ignored_optional_args = []
        if ignored_arg_values is None:
            ignored_arg_values = {}
        if arg_descriptions is None:
            arg_descriptions = {}
        self.ignored_optional_args = ignored_optional_args
        self.ignored_arg_values = ignored_arg_values
        self.arg_descriptions = arg_descriptions
        self.info = json.loads(self.wbt.tool_parameters(self.wbt_name))

    def initial_arguments(self):
        """Get initial arguments for tool.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = []
        for p in self.info['parameters']:
            p_type = p['parameter_type']
            name = get_python_friendly_name(p['name'])
            if name in self.ignored_optional_args or name in self.ignored_arg_values:
                continue
            if name in self.arg_descriptions:
                description = self.arg_descriptions[name]
            else:
                description = p['description'].strip('., ')
            if 'ExistingFile' in p_type or 'NewFile' in p_type:
                file_type = ''
                io_direction = IoDirection.INPUT
                if 'ExistingFile' in p_type:
                    file_type = p_type['ExistingFile']
                elif 'NewFile' in p_type:
                    file_type = p_type['NewFile']
                    io_direction = IoDirection.OUTPUT
                if 'Raster' in file_type:
                    arguments.append(self.raster_argument(name=name, description=description,
                                                          io_direction=io_direction, optional=p['optional'],
                                                          value=p['default_value']))
                elif 'Vector' in file_type or 'RasterAndVector' in file_type:
                    arguments.append(self.coverage_argument(name=name, description=description,
                                                            io_direction=io_direction, optional=p['optional'],
                                                            value=p['default_value']))
                else:
                    file_filter = ''
                    if 'Lidar' in file_type:
                        file_filter = 'LiDAR files (*.las, *.zlidar, *.laz, *.zip)'
                    elif 'Text' in file_type:
                        file_filter = 'Text files (*.txt), All files (*.*)'
                    elif 'Csv' in file_type:
                        file_filter = 'CSV files (*.csv), All files (*.*)'
                    elif 'Dat' in file_type:
                        file_filter = 'Binary data files (*.dat), All files (*.*)'
                    elif 'Html' in file_type:
                        file_filter = 'HTML files (*.html)'
                    arguments.append(self.file_argument(name=name, description=description,
                                                        io_direction=io_direction, optional=p['optional'],
                                                        value=p['default_value'], file_filter=file_filter))
            elif 'Boolean' in p_type:
                value = None if p['default_value'] is None else p['default_value'].lower() in ['true', '1', 't', 'y',
                                                                                               'yes']
                arguments.append(self.bool_argument(name=name, description=description,
                                                    optional=p['optional'], value=value))
            elif 'ExistingFileOrFloat' in p_type or 'Float' in p_type:
                value = None if p['default_value'] is None else float(p['default_value'])
                arguments.append(self.float_argument(name=name, description=description,
                                                     optional=p['optional'], value=value))
            elif 'Integer' in p_type:
                value = None if p['default_value'] is None else int(p['default_value'])
                arguments.append(self.integer_argument(name=name, description=description,
                                                       optional=p['optional'], value=value))
            elif 'Text' in p_type or 'String' in p_type or 'StringOrNumber' in p_type:
                arguments.append(self.string_argument(name=name, description=description,
                                                      optional=p['optional'], value=p['default_value']))
            elif 'OptionList' in p_type:
                arguments.append(self.string_argument(name=name, description=description, optional=p['optional'],
                                                      value=p['default_value'], choices=p_type['OptionList']))
            else:
                self.logger.error(f"Unsupported parameter type: {p_type}.")
        return arguments

    def run(self, arguments):
        """Runs the tool.

        Args:
            arguments (list): A list of the tool's arguments.
        """
        raster_wkt, wbt_args = parse_arguments(arguments, self, self.info, self.ignored_optional_args,
                                               self.ignored_arg_values)
        if self.wbt.run_tool(self.wbt_name, wbt_args) == 1:
            raise RuntimeError(f"Error running {self.name}.")
        set_output_data(raster_wkt, arguments, self, self.info, self.ignored_optional_args, self.ignored_arg_values,
                        True)
