"""WhiteboxTool base class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import json

# 2. Third party modules

# 3. Aquaveo modules
from xms.wbtools.whitebox_tools import WhiteboxTools

# 4. Local modules
from xms.tool.whitebox.whitebox_tool import parse_arguments, set_output_data


def run_wbt_tool(wbt_runner, wbt_name, specified_arg_values, send_outputs_to_xms):
    """Runs the whitebox tool using wbt_runner.

    Args:
        wbt_runner (WhiteboxToolRunner): The whitebox tool runner.
        wbt_name (str): The whitebox tool class name. This is also used as the tool name is no tool_name is defined.
        specified_arg_values (dict): A dict containing python-friendly argument names as the key and their values
         for which the values will be given to the whitebox tool.
        send_outputs_to_xms (bool): Whether the outputs (raster and/or vector data) from running the tool will be
         sent to XMS after the tool is run.
    """
    run_wbt_tool_internal(wbt_runner, wbt_name, specified_arg_values, send_outputs_to_xms)


def run_wbt_tool_internal(wbt_runner, wbt_name, specified_arg_values, send_outputs_to_xms):
    """Runs the whitebox tool using wbt_runner. Call when mocking so we don't have to call run_wbt_tool above.

    Args:
        wbt_runner (WhiteboxToolRunner): The whitebox tool runner.
        wbt_name (str): The whitebox tool class name. This is also used as the tool name is no tool_name is defined.
        specified_arg_values (dict): A dict containing python-friendly argument names as the key and their values
         for which the values will be given to the whitebox tool.
        send_outputs_to_xms (bool): Whether the outputs (raster and/or vector data) from running the tool will be
         sent to XMS after the tool is run.
    """
    wbt_runner.set_info(wbt_name, specified_arg_values, send_outputs_to_xms)
    wbt_runner.run_tool()


class WhiteboxToolRunner:
    """WhiteboxToolRunner class."""

    def __init__(self, tool, wbt_name='', specified_arg_values=None, send_outputs_to_xms=True):
        """Initializes the class.

        Args:
            tool (xms.tool_core.Tool): The tool.
            wbt_name (str): The whitebox tool class name. This is also used as the tool name is no tool_name is defined.
            specified_arg_values (dict): A dict containing python-friendly argument names as the key and their values
             for which the values will be given to the whitebox tool.
            send_outputs_to_xms (bool): Whether the outputs (raster and/or vector data) from running the tool will be
             sent to XMS after the tool is run.
        """
        self.tool = tool
        self.wbt = WhiteboxTools()
        self.wbt.set_compress_rasters(False)
        self.wbt.work_dir = ''
        self.wbt.set_default_callback(self.tool.logger.info)
        self.wbt_name = None
        self.specified_arg_values = None
        self._send_outputs_to_xms = None
        self.info = None
        if wbt_name:
            self.set_info(wbt_name, specified_arg_values, send_outputs_to_xms)

    def set_info(self, wbt_name, specified_arg_values, send_outputs_to_xms=True):
        """Sets the info for the runner.

        Args:
            wbt_name (str): The whitebox tool class name. This is also used as the tool name is no tool_name is defined.
            specified_arg_values (dict): A dict containing python-friendly argument names as the key and their values
             for which the values will be given to the whitebox tool.
            send_outputs_to_xms (bool): Whether the outputs (raster and/or vector data) from running the tool will be
             sent to XMS after the tool is run.
        """
        self.wbt_name = wbt_name
        if specified_arg_values is None:
            specified_arg_values = {}
        self.specified_arg_values = specified_arg_values
        self._send_outputs_to_xms = send_outputs_to_xms
        self.info = json.loads(self.wbt.tool_parameters(self.wbt_name))

    def run_tool(self):
        """Runs the tool."""
        arguments = []
        raster_wkt, wbt_args = parse_arguments(arguments, self.tool, self.info, [], self.specified_arg_values)
        if self.wbt.run_tool(self.wbt_name, wbt_args) == 1:
            raise RuntimeError(f"Error running {self.wbt_name}.")
        set_output_data(raster_wkt, arguments, self.tool, self.info, [], self.specified_arg_values,
                        self._send_outputs_to_xms)
