"""BoolArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, IoDirection


class BoolArgument(Argument):
    """String tool argument."""
    def __init__(
        self,
        name: str,
        description: str = '',
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[bool] = None,
        hide: bool = False
    ):
        """Construct a bool argument.

        Args:
            name (str): Python friendly argument name.
            description (str): User friendly description of the argument.
            io_direction (IoDirection): IO Direction of the argument (input or output).
            optional (bool): Is the argument optional?
            value (Optional[bool]): Default value.
            hide (bool): Should the argument be hidden (True) or visible?

        Returns:
            (BoolArgument): A bool argument.
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        if self.io_direction == IoDirection.OUTPUT:
            self.optional = True  # output doesn't need to be specified

    def _get_type(self) -> str:
        """Get a string representing the argument type (string).

        Returns:
            (str): The argument type.
        """
        return 'bool'

    def _set_value(self, value: Optional[bool]) -> None:
        """Set the argument value.

        Args:
            value (obj): The new argument value.
        """
        if value is None:
            self._value = value
        else:
            self._value = bool(value)

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        if self.io_direction == IoDirection.OUTPUT:
            return None
        interface_info = {'type': 'Boolean'}
        interface_info = interface_info | super().get_interface_info()
        if interface_info['value'] is None:
            interface_info['value'] = False
        return interface_info

    def to_dict(self) -> dict:
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'BoolArgument'}
        super_values = super(BoolArgument, self).to_dict()
        values.update(super_values)
        return values
