"""ColorArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.tool_core.argument import Argument, IoDirection


class ColorArgument(Argument):
    """Color tool argument."""
    def __init__(
        self,
        name: str,
        description: str = '',
        io_direction: Optional[IoDirection] = IoDirection.INPUT,
        optional: Optional[bool] = False,
        value: Optional[tuple[int, int, int]] = None,
        hide: Optional[bool] = False,
    ):
        """Construct a dataset argument.

        Args:
            name (Optional[str]): Python friendly argument name.
            description (Optional[str]): User friendly description of the argument.
            io_direction (Optional[IoDirection]): IO Direction of the argument (input or output).
            optional (Optional[bool]): Is the argument optional?
            value (Optional[int]): Default value.
            hide (Optional[bool]): Should the argument be hidden (True) or visible?
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        if optional is None:
            raise ValueError("Color arguments can't be optional.")

    def _get_type(self) -> str:
        """Get a string representing the argument type (color).

        Returns:
            (str): The argument type.
        """
        return 'color'

    def _set_value(self, value: Optional[tuple[int, int, int]]) -> None:
        """Set the argument value.

        Args:
            value (Optional[int]): The new argument value.
        """
        self._value = None if value is None else value

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        if self.io_direction == IoDirection.OUTPUT:
            return None
        interface_info = {'type': 'Color'}
        interface_info = interface_info | super().get_interface_info()
        if interface_info['value'] is None:
            interface_info['value'] = (0, 0, 0)
        return interface_info

    def validate(self) -> str:
        """Validate the argument.

        Returns:
            (str): An error string if invalid or None.
        """
        if self.value is not None:
            r, g, b = self.value
            # check all r, g, b values are between 0 and 255
            if r < 0 or r > 255:
                return 'R value must be between 0 and 255.'
            if g < 0 or g > 255:
                return 'G value must be between 0 and 255.'
            if b < 0 or b > 255:
                return 'B value must be between 0 and 255.'
        return None

    def to_dict(self) -> dict:
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'ColorArgument'}
        super_values = super(ColorArgument, self).to_dict()
        values.update(super_values)
        return values
