"""StringArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, IoDirection


class StringArgument(Argument):
    """String tool argument."""
    def __init__(
        self,
        name: str,
        description: str = '',
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: str | None = None,
        hide: bool = False,
        choices: list[str] = None,
        file: bool = False
    ):
        """Construct a string argument.

        Args:
            name (str): Python friendly argument name.
            description (str): User friendly description of the argument.
            io_direction (IoDirection): IO Direction of the argument (input or output).
            optional (bool): Is the argument optional?
            value (Optional[string]): Default value.
            hide (bool): Should the argument be hidden (True) or visible?
            choices (list[string]): List of potential argument values.
            file (bool):  False if not using a file, else use a file selector.

        Returns:
            (StringArgument): A floating point argument.
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        self.choices = choices
        self.file = file
        if self.io_direction == IoDirection.OUTPUT:
            self.optional = True  # output doesn't need to be specified

    def _get_type(self):
        """Get a string representing the argument type (string).

        Returns:
            (str): The argument type.
        """
        return 'string'

    def _set_value(self, value):
        """Set the argument value.

        Args:
            value (obj): The new argument value.
        """
        self._value = None if value is None else str(value)

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        if self.io_direction == IoDirection.OUTPUT:
            return None
        if self.choices is not None:
            interface_info = {
                'type': 'StringSelector',
                'choices': self.choices.copy()
            }
        elif self.file:
            interface_info = {
                'type': 'SelectFile',
                'filter': 'All Files (*.*)'
            }
        else:
            interface_info = {'type': 'String'}
        interface_info = interface_info | super().get_interface_info()
        if interface_info['value'] is None:
            if self.choices is not None:
                interface_info['value'] = interface_info['choices'][0]
            else:
                interface_info['value'] = ''
        return interface_info

    def to_dict(self):
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'StringArgument'}
        super_values = super(StringArgument, self).to_dict()
        values.update(super_values)
        self._add_key_value('choices', self.choices, values)
        self._add_key_value('file', self.file, values)
        return values

    def validate(self):
        """Validate the argument.

        Returns:
            (str): An error string if invalid or None.
        """
        result = super().validate()
        if result is not None:
            return result

        if self.choices and self.value is not None and self.show and self.value not in self.choices:
            return 'Argument value not in available choices.'

        if self.file and not self.optional and self.io_direction == IoDirection.INPUT:
            if not os.path.isfile(str(self.value)):
                return 'Could not find file.'
        return None
