"""Table classes."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import asdict, dataclass
from datetime import datetime
import sys

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules

# 4. Local modules


class TableDefinition:
    """Defines the table column types."""
    def __init__(self, column_types, fixed_row_count: int | None = None) -> None:
        """Initializes the class.

        Args:
            column_types (list[Type(ColumnType)]): List of column types.
            fixed_row_count (int | None): If given, the table will have the number of rows fixed at fixed_row_count.
        """
        self.column_types = column_types
        self.fixed_row_count = fixed_row_count

    def to_dict(self):
        """Returns a dict of the data."""
        column_types = [column_type.to_dict() for column_type in self.column_types]
        return {'column_types': column_types, 'fixed_row_count': self.fixed_row_count}

    def to_pandas(self, json_str: str = '', rows=None) -> pd.DataFrame:
        """Returns a pandas DataFrame given the table definition and, optionally, values in the json_str or in the rows.

        See https://stackoverflow.com/questions/36462257

        Args:
            json_str (str): The table as a json string created from pd.DataFrame.to_json(orient='table').
            rows (list[list[Any]]): 2D list of values, row-wise.

        Returns:
            (pd.DataFrame): The DataFrame.
        """
        if json_str and rows:
            raise ValueError('You cannot pass both "json_str" and "rows"')
        if json_str:  # Create the DataFrame from the values in the json_str and the dtypes in TableDefinition
            df = pd.read_json(json_str, orient='table')
            df = df.astype(dtype={column_type.header: column_type.dtype for column_type in self.column_types})
        elif rows:  # Create the DataFrame from the row values and the dtypes in TableDefinition
            columns = [list(x) for x in zip(*rows)]  # Transpose from rows to columns
            df = pd.DataFrame({column_type.header: columns[i] for i, column_type in enumerate(self.column_types)})
            df.index += 1  # Make index start at 1
            df = df.astype(dtype={column_type.header: column_type.dtype for column_type in self.column_types})
        else:  # Create DataFrame with the right columns and dtypes but no values
            df_dict = {column_type.header: pd.Series(dtype=column_type.dtype) for column_type in self.column_types}
            df = pd.DataFrame(df_dict)
        return df

    @classmethod
    def from_dict(cls, d) -> 'TableDefinition':
        """Returns an instance of the class given a dict.

        Args:
            d: The dict.

        Returns:
            (TableDefinition): The object.
        """
        column_types = []
        column_type_dicts = d.get('column_types', [])
        for column_type_dict in column_type_dicts:
            if column_type_dict['type'] == 'IntColumnType':
                column_type = IntColumnType.from_dict(column_type_dict)
            elif column_type_dict['type'] == 'FloatColumnType':
                column_type = FloatColumnType.from_dict(column_type_dict)
            elif column_type_dict['type'] == 'InputFileColumnType':
                column_type = InputFileColumnType.from_dict(column_type_dict)
            elif column_type_dict['type'] == 'StringColumnType':
                column_type = StringColumnType.from_dict(column_type_dict)
            elif column_type_dict['type'] == 'ChoicesColumnType':
                column_type = ChoicesColumnType.from_dict(column_type_dict)
            elif column_type_dict['type'] == 'DateTimeColumnType':
                column_type = DateTimeColumnType.from_dict(column_type_dict)
            else:
                column_type_str = column_type_dict['type']
                raise ValueError(f'Unsupported column type "{column_type_str}".')
            column_types.append(column_type)

        fixed_row_count = d.get('fixed_row_count')
        return TableDefinition(column_types, fixed_row_count)


@dataclass
class ColumnType:
    """Base column type."""
    header: str = ''
    tool_tip: str = ''
    enabled: bool = True
    dtype: str = ''


@dataclass
class StringColumnType(ColumnType):
    """String column type.

    If choices is specified as an int, the combobox choices are defined in the column with that index. This method
    is used when different lists of choices are needed for each row. The choice column should be a ChoicesColumnType
    and will be hidden.
    """
    dtype: str = 'str'
    default: str = ''
    choices: list[str] | int | None = None

    def to_dict(self):
        """Returns a dict of the data."""
        return {'type': 'StringColumnType', **asdict(self)}

    @classmethod
    def from_dict(cls, d) -> 'StringColumnType':
        """Returns an instance of the class given a dict.

        Args:
            d: The dict.

        Returns:
            (StringColumnType): The object.
        """
        return StringColumnType(
            header=d.get('header', ''),
            tool_tip=d.get('tool_tip', ''),
            enabled=d.get('enabled', True),
            dtype=d.get('dtype', 'str'),
            default=d.get('default', ''),
            choices=d.get('choices', None)
        )


@dataclass
class ChoicesColumnType(ColumnType):
    """Special hidden column type holding choices for another column.

    Column containing lists of choices to be used in the comboboxes of another column.
    """
    dtype: str = 'object'
    default: list | None = None

    def to_dict(self):
        """Returns a dict of the data."""
        return {'type': 'ChoicesColumnType', **asdict(self)}

    @classmethod
    def from_dict(cls, d) -> 'ChoicesColumnType':
        """Returns an instance of the class given a dict.

        Args:
            d: The dict.

        Returns:
            (StringColumnType): The object.
        """
        return ChoicesColumnType(
            header=d.get('header', ''),
            tool_tip=d.get('tool_tip', ''),
            enabled=d.get('enabled', True),
            dtype=d.get('dtype', 'object'),
            default=d.get('default', []),
        )


@dataclass
class IntColumnType(ColumnType):
    """Int column type."""
    dtype: str = 'int'
    default: int = 0
    low: int | None = None
    high: int | None = None
    spinbox: bool = False  # You can choose spinbox or checkbox but not both
    checkbox: bool = False

    def to_dict(self):
        """Returns a dict of the data."""
        return {'type': 'IntColumnType', **asdict(self)}

    @classmethod
    def from_dict(cls, d) -> 'IntColumnType':
        """Returns an instance of the class given a dict.

        Args:
            d: The dict.

        Returns:
            (IntColumnType): The object.
        """
        return IntColumnType(
            header=d.get('header', ''),
            tool_tip=d.get('tool_tip', ''),
            enabled=d.get('enabled', True),
            dtype=d.get('dtype', 'int'),
            default=d.get('default', 0.0),
            low=d.get('low', sys.float_info.min),
            high=d.get('high', sys.float_info.max),
            spinbox=d.get('spinbox', True),
            checkbox=d.get('checkbox', False),
        )


@dataclass
class FloatColumnType(ColumnType):
    """Float column type."""
    dtype: str = 'float'
    default: float = 0.0
    low: float = -sys.float_info.max
    high: float = sys.float_info.max

    def to_dict(self):
        """Returns a dict of the data."""
        return {'type': 'FloatColumnType', **asdict(self)}

    @classmethod
    def from_dict(cls, d) -> 'FloatColumnType':
        """Returns an instance of the class given a dict.

        Args:
            d: The dict.

        Returns:
            (FloatColumnType): The object.
        """
        return FloatColumnType(
            header=d.get('header', ''),
            tool_tip=d.get('tool_tip', ''),
            enabled=d.get('enabled', True),
            dtype=d.get('dtype', 'int'),
            default=d.get('default', 0.0),
            low=d.get('low', sys.float_info.min),
            high=d.get('high', sys.float_info.max),
        )


@dataclass
class InputFileColumnType(ColumnType):
    """Input file column type."""
    dtype: str = 'str'
    default: str = ''
    file_filter: str = '*.*'
    default_suffix: str = ''

    def to_dict(self):
        """Returns a dict of the data."""
        return {'type': 'InputFileColumnType', **asdict(self)}

    @classmethod
    def from_dict(cls, d) -> 'InputFileColumnType':
        """Returns an instance of the class given a dict.

        Args:
            d: The dict.

        Returns:
            (InputFileColumnType): The object.
        """
        return InputFileColumnType(
            header=d.get('header', ''),
            tool_tip=d.get('tool_tip', ''),
            enabled=d.get('enabled', True),
            dtype=d.get('dtype', 'int'),
            default=d.get('default', 0.0),
            file_filter=d.get('file_filter', ''),
            default_suffix=d.get('default_suffix', '')
        )


@dataclass
class DateTimeColumnType(ColumnType):
    """Date/Time column type."""
    dtype: str = 'datetime64[ns]'
    default: datetime = datetime(1970, 1, 1, 0, 0)

    def to_dict(self):
        """Returns a dict of the data."""
        data_dict = asdict(self)
        data_dict['default'] = self.default.isoformat()
        return {'type': 'DateTimeColumnType', **data_dict}

    @classmethod
    def from_dict(cls, d) -> 'DateTimeColumnType':
        """Returns an instance of the class given a dict.

        Args:
            d: The dict.

        Returns:
            (DateTimeColumnType): The object.
        """
        default_str = d.get('default', '1970-01-01 00:00:00')
        default = pd.to_datetime(default_str)
        return DateTimeColumnType(
            header=d.get('header', ''),
            tool_tip=d.get('tool_tip', ''),
            enabled=d.get('enabled', True),
            dtype=d.get('dtype', 'datetime64[ns]'),
            default=default,
        )

# I only made the ColumnType classes that I needed. Feel free to add more here. -MJK
