"""TimestepArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, IoDirection


class TimestepArgument(Argument):
    """Timestep tool argument."""
    def __init__(
        self,
        data_handler,
        name: str,
        description: str = '',
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: str = None,
        hide: bool = False
    ):
        """Construct a timstep argument.

        Args:
            data_handler (DataHandler): Data handler to retrieve timestep's dataset.
            name (str): Python friendly argument name.
            description (str): User friendly description of the argument.
            io_direction (IoDirection): IO Direction of the argument (input or output).
            optional (bool): Is the argument optional?
            value (Optional[str]): Default value.
            hide (bool): Should the argument be hidden (True) or visible?
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        self.data_handler = data_handler
        self.choices = [Argument.NONE_SELECTED]
        if io_direction == IoDirection.OUTPUT:
            raise RuntimeError('Output timestep not allowed.')

    def _get_type(self):
        """Get a string representing the argument type.

        Returns:
            (str): The argument type.
        """
        return 'timestep'

    def _set_value(self, value):
        """Set the argument value.

        Args:
            value (str): The new argument value.
        """
        self._value = None if value is None else str(value)

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        choices = self.choices.copy()
        interface_info = {'type': 'StringSelector', 'choices': choices}
        interface_info = interface_info | super().get_interface_info()
        if self.value not in choices:
            interface_info['value'] = Argument.NONE_SELECTED
        return interface_info

    def to_dict(self):
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'TimestepArgument'}
        super_values = super(TimestepArgument, self).to_dict()
        values.update(super_values)
        return values

    def enable_timestep(self, dataset_argument):
        """Enable timesteps argument based on dataset.

        Args:
            dataset_argument (DatasetArgument): The dataset argument.
        """
        dataset = self.data_handler.get_input_dataset(dataset_argument.value)
        if dataset is None or dataset.num_times <= 1:
            self.hide = True
        else:
            # fill in timestep list if more than one timestep
            self.hide = False
            choices = [Argument.NONE_SELECTED]
            for time_idx in range(dataset.num_times):
                choices.append(str(time_idx + 1))
            if choices != self.choices:
                self.choices = choices

    def validate_timestep(self, dataset_argument, errors):
        """Validate dataset and timestep arguments.

        Args:
            dataset_argument (DatasetArgument): The dataset argument.
            errors (dict): Dictionary of errors.
        """
        if dataset_argument.text_value == '':
            errors[self.name] = 'Input dataset not specified.'
        else:
            dataset = self.data_handler.get_input_dataset(dataset_argument.text_value)
            if dataset is None:
                errors[dataset_argument.name] = 'Invalid input dataset.'
            elif dataset.num_times > 1:
                # Validate time step is specified
                timestep = self.get_timestep(dataset_argument)
                if timestep is None:
                    errors[self.name] = 'Input time not specified.'

    def get_timestep(self, dataset_argument):
        """Get the selected timestep.

        Args:
            dataset_argument (DatasetArgument): The dataset argument.

        Returns:
            (int): The selected timestep index or -1 for none selected.
        """
        dataset = self.data_handler.get_input_dataset(dataset_argument.text_value)
        if dataset is None:
            return None
        elif dataset.num_times == 1:
            return 0
        elif self.value is None:
            return None
        try:
            return int(self.value) - 1
        except ValueError:
            return None
