"""OutputPointsComponent class."""
# 1. Standard python modules
import os

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.components import output_points_component_display as op_disp
from xms.tuflowfv.components.tuflowfv_component import TuflowFvComponent
from xms.tuflowfv.data.output_points_data import OutputPointsData


__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"


class OutputPointsComponent(TuflowFvComponent):
    """A hidden Dynamic Model Interface (DMI) component for the TUFLOWFV Output Points coverage."""

    def __init__(self, main_file):
        """Initializes the component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = OutputPointsData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.class_name = 'OutputPointsComponent'
        self.module_name = 'xms.tuflowfv.components.output_points_component'
        self.tree_commands = [
            ('Display Options...', 'open_display_options'),
        ]
        self.point_commands = [('Point Attributes...', 'open_assign_point')]

        comp_dir = os.path.dirname(self.main_file)
        self.disp_opts_files = [os.path.join(comp_dir, op_disp.OUTPUT_POINT_JSON)]
        display_helper = op_disp.OutputPointsComponentDisplay(self)
        display_helper.ensure_display_option_files_exist()

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        messages = []
        action_requests = [self.get_display_options_action()]
        return messages, action_requests

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            tuple(str, list, list):
                - new_main_file (str): Path to the new component's mainfile
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            helper = op_disp.OutputPointsComponentDisplay(self)
            helper.update_display_options_file(new_main_file, new_path)

        return new_main_file, messages, action_requests

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (Query): Object for communicating with XMS
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.

        Returns:
            Empty message and ActionRequest lists

        """
        helper = op_disp.OutputPointsComponentDisplay(self, query)
        helper.get_initial_display_options(query, params)
        return [], []

    def open_assign_point(self, query, params, win_cont):
        """Opens the Assign Pipe dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the
                    message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        helper = op_disp.OutputPointsComponentDisplay(self, query)
        return helper.assign_attributes(params=params, parent=win_cont)

    def open_display_options(self, query, params, win_cont):
        """Shows the display options dialog.

        Args:
            query (Query): Object for communicating with XMS
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        helper = op_disp.OutputPointsComponentDisplay(self, query)
        return helper.display_options(win_cont, self.cov_uuid)

    def handle_merge(self, merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (list(tuple)): tuple containing:
                - main_file (str): The absolute path to the main file of the old component this
                  component is being merged from.
                - id_files (dict): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                  Each value is a tuple that may have two absolute file paths or none. The first
                  file is for the ids in XMS on the coverage. The second file contains the ids the
                  old component used for those objects. Both id files should be equal in length.
                  This dictionary is only applicable if the component derives from
                  CoverageComponentBase.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        return [], []
