"""MaterialData class."""
# 1. Standard python modules

# 2. Third party modules
import pandas as pd
import pkg_resources

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem

# 4. Local modules

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


DEFAULT_MATERIAL_COLORS = [
    (255, 0, 0),  # red
    (0, 0, 255),  # blue
    (0, 255, 51),  # green
    (255, 204, 0),  # yellow
    (0, 204, 255),  # cyan
    (255, 0, 204),  # magenta
    (153, 0, 255),  # purple
    (255, 102, 0),  # orange
    (51, 153, 51),  # dark green
    (153, 102, 0),  # brown
    (102, 153, 153),  # grey blue
    (153, 51, 51),  # dark red
    (255, 153, 255),  # light magenta
]


class MaterialData(XarrayBase):
    """Class for storing the TUFLOWFV Material properties (Manning's N and sediment)."""
    UNASSIGNED_MAT = 0  # The unassigned material id (and row in material list)
    # constants for columns in the materials dataset
    COL_ID = 0
    COL_COLOR = 1
    COL_NAME = 2
    COL_ACTIVE = 3
    COL_OVERRIDE_BOTTOM_ROUGHNESS = 4
    COL_BOTTOM_ROUGHNESS = 5
    COL_ADVANCED = 6

    def __init__(self, filename):
        """Constructor.

        Args:
            filename (str): file name
        """
        super().__init__(filename)
        self.info.attrs['FILE_TYPE'] = 'TUFLOWFV_MATERIAL_DATA'
        if 'cov_uuid' not in self.info.attrs:
            self.info.attrs['cov_uuid'] = ''  # gets set later
        if 'display_uuid' not in self.info.attrs:
            self.info.attrs['display_uuid'] = ''
        self._materials = None  # materials datasets
        _ = self.materials  # Load all data from disk.
        self.close()

    @property
    def materials(self):
        """Get the material component data parameters.

        Returns:
            xarray.Dataset: The material list dataset

        """
        if self._materials is None:
            self._materials = self.get_dataset('materials', False)
            if self._materials is None:
                self._materials = self._default_materials()
        return self._materials

    @materials.setter
    def materials(self, dset):
        """Setter for the materials dataset."""
        if dset:
            self._materials = dset

    @staticmethod
    def default_data_dict():
        """Returns a dict of default values for a new material."""
        return {
            'id': [0],
            'color': ['0 0 0 1'],
            'name': ['unassigned'],
            'inactive': [0],
            'override_bottom_roughness': [0],
            'bottom_roughness': [0.05],
            'override_horizontal_eddy_viscosity': [0],
            'horizontal_eddy_viscosity': [0.2],
            'override_horizontal_eddy_viscosity_limits': [0],
            'horizontal_eddy_viscosity_minimum': [0.0],
            'horizontal_eddy_viscosity_maximum': [99999.0],
            'override_vertical_eddy_viscosity_limits': [0],
            'vertical_eddy_viscosity_minimum': [1.0e-04],
            'vertical_eddy_viscosity_maximum': [99999.0],
            'override_horizontal_scalar_diffusivity': [0],
            'horizontal_scalar_diffusivity': [0.2],
            'override_horizontal_scalar_diffusivity_limits': [0],
            'horizontal_scalar_diffusivity_minimum': [1.0],
            'horizontal_scalar_diffusivity_maximum': [99999.0],
            'override_vertical_scalar_diffusivity_limits': [0],
            'vertical_scalar_diffusivity_minimum': [1.0e-06],
            'vertical_scalar_diffusivity_maximum': [99999.0],
            'override_bed_elevation_limits': [0],
            'bed_elevation_minimum': [-10.0],
            'bed_elevation_maximum': [9999.0],
            'spatial_reconstruction': [0],
        }

    def _default_materials(self):
        """Creates a default material dataset.

        Returns:
            xarray.Dataset: The material list dataset
        """
        self.info.attrs['export_format'] = '2dm'
        return pd.DataFrame(self.default_data_dict()).to_xarray()

    def add_materials(self, mat_data):
        """Adds the materials from mat_data to this instance of MaterialData.

        Args:
            mat_data (MaterialData): another Material Data instance

        Returns:
            dict: The old ids of the mat_data as key and the new ids as the data
        """
        mat_df = self.materials.to_dataframe()
        mat_names = set(mat_df['name'].to_list())
        mat_data_df = mat_data.materials.to_dataframe()
        old_to_new_ids = {0: 0}  # UNASSIGNED_MAT must always be in the dict
        next_mat_id = max(mat_df['id']) + 1
        next_idx = len(mat_df)
        for i in range(1, len(mat_data_df)):  # start at 1 to skip UNASSIGNED_MAT
            mat_id = mat_data_df.iloc[i, MaterialData.COL_ID]
            new_mat_id = next_mat_id
            next_mat_id += 1
            old_to_new_ids[mat_id] = new_mat_id
            mat_data_df.iloc[i, MaterialData.COL_ID] = new_mat_id
            mat_name = mat_data_df.iloc[i, MaterialData.COL_NAME]
            cnt = 1
            orig_mat_name = mat_name
            while mat_name in mat_names:
                mat_name = f'{orig_mat_name} ({cnt})'
                cnt += 1
            mat_data_df.iloc[i, MaterialData.COL_NAME] = mat_name
            mat_df.loc[next_idx] = mat_data_df.values[i]
            next_idx += 1
        self._materials = mat_df.to_xarray()
        return old_to_new_ids

    def commit(self):
        """Save in memory datasets to the NetCDF file."""
        self.info.attrs['VERSION'] = pkg_resources.get_distribution('xmstuflowfv').version
        filesystem.removefile(self._filename)  # Always vacuum the file. Materials dataset is small enough.
        super().commit()
        self.materials.to_netcdf(self._filename, group='materials', mode='a')

    def close(self):
        """Closes the H5 file and does not write any data that is in memory."""
        super().close()
        if self._materials is not None:
            self._materials.close()
