"""Defines the structure BCs."""

__copyright__ = '(C) Copyright Aquaveo 2022'
__license__ = 'All rights reserved'

# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules
import xms.gmi.data.generic_model as gm
from xms.guipy.data.target_type import TargetType

# 4. Local modules


STRUCT_INITIAL_ARC_ATT_ID_FILE = 'arc_atts.ids'
STRUCT_INITIAL_ARC_COMP_ID_FILE = 'arc_comp.ids'
STRUCT_INITIAL_POLY_ATT_ID_FILE = 'poly_atts.ids'
STRUCT_INITIAL_POLY_COMP_ID_FILE = 'poly_comp.ids'
ENERGY_LOSS_FUNCTIONS = ['Coefficient', 'Table']
CULVERT_TYPES = ['Circular', 'Rectangular', 'Gated circular (unidirectional)', 'Gated rectangular (unidirectional)']
SET_TYPE_NLSWE = 'Nlswe'
SET_TYPE_CULVERT = 'Culvert'


class StructureDef:
    """Class to generate gmi model definition for structures."""
    def __init__(self, num_selected, target_type, single_select_set, feature1_id, feature2_id):
        """Constructor.

        Args:
            num_selected (int): The number of features selected
            target_type (TargetType): The feature type of the selection
            single_select_set (bool): True if num_selected == 1 but the selected feature is part of a valid set
            feature1_id (int): Att id of the first feature in the set
            feature2_id (int): Att id of the second feature in the set, if it exists
        """
        self.num_selected = num_selected
        self.target_type = target_type
        self.single_select_set = single_select_set
        self.feature1_id = feature1_id
        self.feature2_id = feature2_id
        self.parameters = None

    @staticmethod
    def _add_culvert_widgets(culvert):
        """Add weir specific widgets.

        Args:
            culvert (Group): The culvert group
        """
        culvert.add_option('type', label='Type', options=CULVERT_TYPES, default=CULVERT_TYPES[0])
        culvert.add_boolean('ignore', label='Ignore', default=False)
        culvert.add_float('len_or_ana', label='Length', default=0.0)
        culvert.add_float('n_or_n_f', label='Friction', default=0.0)
        culvert.add_float('us_invert', label='Upstream Invert Level', default=0.0)
        culvert.add_float('ds_invert', label='Downstream Invert Level', default=0.0)
        culvert.add_float('form_loss', label='Form Loss Coefficient', default=0.0)
        culvert.add_integer('pblockage', label='Percent Blockage', default=0, low=0, high=100)
        culvert.add_float('width_or_dia', label='Width/Diameter', default=0.0)
        culvert.add_float('height_or_wf', label='Height', default=0.0)
        culvert.add_integer('number_of', label='Number of Barrels', default=0, low=0)
        culvert.add_float('height_cont', label='Height Contraction Coefficient', default=0.6)
        culvert.add_float('width_cont', label='Width Contraction Coefficient', default=1.0)
        culvert.add_float('entry_loss', label='Entry Loss Coefficient', default=0.5)
        culvert.add_float('exit_loss', label='Exit Loss Coefficient', default=0.5)

    def _add_weir_widgets(self):
        """Add weir specific widgets."""
        weir = self.parameters.add_group(label='Weir', group_name='Weir', allow_interior=False)
        weir.add_text('name', label='Name', default='')
        weir.add_float('weir_z', label='Elevation', default=0.0)
        weir.add_boolean('elevation_is_dz', label='Elevation is dz', default=True)
        weir.add_boolean('define_weir', label='Override weir defaults', default=False)
        # Widgets should only show up if we are not using the default weir definition
        parent = [self.parameters.name, 'Weir', 'define_weir']
        flags = {True: True, False: False}
        param = weir.add_float('weir_cw', label='Weir CW', default=1.705)
        param.add_dependency(parent=parent, flags=flags)
        param = weir.add_float('weir_ex', label='Weir EX', default=1.5)
        param.add_dependency(parent=parent, flags=flags)
        param = weir.add_float('weir_a', label='Weir A', default=8.55)
        param.add_dependency(parent=parent, flags=flags)
        param = weir.add_float('weir_b', label='Weir B', default=0.556)
        param.add_dependency(parent=parent, flags=flags)
        param = weir.add_float('weir_csfm', label='Weir CSFM', default=0.7)
        param.add_dependency(parent=parent, flags=flags)

    def _add_set_widgets(self, set_type):
        """Add common widgets.

        Args:
            set_type (str): Name of the set group. One of the SET_TYPE_* constants.
        """
        # If a polygon Bridge, set `is_active` to True so Bridge is the default type.
        is_active = self.target_type == TargetType.polygon and set_type == SET_TYPE_NLSWE
        group = self.parameters.add_group(label=set_type, group_name=set_type, allow_interior=False,
                                          is_active=is_active)
        group.add_text('name', label=f'Name 1 (Feature ID: {self.feature1_id})', default='')
        if self.num_selected == 2 or self.single_select_set:  # No `Name 2` widget if assigning to a single arc.
            group.add_text('name2', label=f'Name 2 (Feature ID: {self.feature2_id})', default='')
            group.add_option('upstream', label="Upstream", options=['Name 1', 'Name 2'], default='Name 1')
        if set_type == SET_TYPE_CULVERT:
            self._add_culvert_widgets(group)  # Add culvert-specific widgets if needed.

        # Show form loss and optional files if energy loss function is 'Coefficient'.
        if set_type == SET_TYPE_NLSWE:  # Doesn't apply to culverts.
            group.add_option('energy_loss_function', label='Energy loss function', options=ENERGY_LOSS_FUNCTIONS,
                             default=ENERGY_LOSS_FUNCTIONS[0])
            parent = [self.parameters.name, set_type, 'energy_loss_function']
            flags = {'Coefficient': True, 'Table': False}

            p = group.add_float('form_loss_coefficient', label='Energy loss coefficient', default=0.0)
            p.add_dependency(parent=parent, flags=flags)

            # Blockage and width files are mutually exclusive.
            p = group.add_option('width_or_blockage', label='Blockage or width file',
                                 options=['None', 'Blockage', 'Width'], default='None')
            p.add_dependency(parent=parent, flags=flags)
            sub_parent = [self.parameters.name, set_type, 'width_or_blockage']

            s = group.add_input_file('blockage_file', label='Blockage file', file_filter='CSV (*.csv);;All types (*.*)',
                                     default='')
            s.add_dependency(parent=sub_parent, flags={'None': False, 'Blockage': True, 'Width': False})

            s = group.add_input_file('width_file', label='Width file', file_filter='CSV (*.csv);;All types (*.*)',
                                     default='')
            s.add_dependency(parent=sub_parent, flags={'None': False, 'Blockage': False, 'Width': True})

            # Energy loss file is only needed if the energy loss function is 'Table'.
            p = group.add_input_file('energy_loss_file', label='Energy loss file',
                                     file_filter='CSV (*.csv);;All types (*.*)', default='')
            p.add_dependency(parent=[self.parameters.name, set_type, 'energy_loss_function'],
                             flags={'Coefficient': False, 'Table': True})

        # Add orientation widgets for zone structures.
        # group.add_float('zone_inlet_orientation', label='Zone inlet orientation', default=0.0)
        # group.add_float('zone_outlet_orientation', label='Zone outlet orientation', default=0.0)

    def structure_def(self):
        """Gets a generic model definition for structure data.

        Returns:
            (GenericModel): The gmi data definition
        """
        model = gm.GenericModel(exclusive_arc_conditions=True)
        self.parameters = model.arc_parameters
        if self.target_type == TargetType.arc:
            self._add_weir_widgets()
        self._add_set_widgets(SET_TYPE_NLSWE)  # Can have a bridge defined by a single arc.
        if self.num_selected == 2 or self.single_select_set:
            self._add_set_widgets(SET_TYPE_CULVERT)
        return model
