"""Worker thread for the TUFLOWFV .fvc file importer."""
# 1. Standard python modules
import logging
import os

# 2. Third party modules
from PySide2.QtCore import Signal

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules
from xms.tuflowfv.file_io.control_reader import ControlReader
from xms.tuflowfv.file_io.io_util import read_multi_sim_file


class ControlImportWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing a TUFLOWFV control input file."""
    processing_finished = Signal()

    def __init__(self, parent, filenames):
        """Construct the worker.

        Args:
            parent (QWidget): The Qt parent
            filenames (list[str]): The .fvc control files to import
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.reader = None
        self._filenames = filenames

    def _do_work(self):
        """Thread runner that imports the .fvc file."""
        try:
            self.reader = ControlReader(self._filenames)
            self.reader.read()
        except Exception as e:
            logging.getLogger('xms.tuflowfv').error(f'Error(s) encountered while reading TUFLOWFV control file: {str(e)}')


def import_simulation_with_feedback(single_sim):
    """Driver for the .fvc import script with a feedback dialog.

    Args:
        single_sim (bool): True if we are exporting a single simulation

    Returns:
        Query: The Query used for testing
    """
    xms_dlg.ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    if XmEnv.xms_environ_running_tests() != 'TRUE':
        parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
        parent = xms_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = win_gui.create_and_connect_raise_timer(main_hwnd, parent)  # Keep the timer in scope
    else:
        parent = None
        main_hwnd = None

    query = Query()
    filename = query.read_file
    if single_sim:
        filenames = [os.path.normpath(filename)]  # Make sure as .fvc filepath is normalized so it can be used in maps
    else:
        filenames = read_multi_sim_file(filename)
    worker = ControlImportWorkerThread(parent, filenames)
    display_text = {
        'title': 'Reading TUFLOWFV Simulation',
        'working_prompt': 'Reading TUFLOWFV control file. Please wait...',
        'error_prompt': 'Error(s) encountered reading simulation file. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered reading simulation file. Review log output for more details.',
        'success_prompt': 'Successfully read TUFLOWFV simulation.',
        'note': '',
        'auto_load': 'Auto load data into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, parent)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():  # and not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.reader.send(query)  # Send data back to XMS
    if parent is not None:
        win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
    return query
