"""Tracks imported data when reading multiple simulations so we don't duplicate data."""
# 1. Standard python modules
import logging
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class DuplicateDataChecker:
    """Class to track the data we have imported during a multi simulation read."""

    def __init__(self):
        """Constructor."""
        self._logger = logging.getLogger('xms.tuflowfv')
        self._imported_files = {}  # {uuid: abs_path}  - files we have already imported
        self._2dm_bcs = {}  # {filename: [uuid]}
        self._2dm_mats = {}  # {filename: [uuid]}
        self._materials = {}  # {uuid: {attr: value}}  - Materials coverages we have already imported
        self._bcs = {}  # {uuid: {attr: value}}  - BC coverages we have already imported
        self._output_points = {}  # {uuid: {attr: value}}  - Output Points coverages we have already imported

    def check_for_imported_file(self, abs_path):
        """Check if we have already imported a file.

        Args:
            abs_path (str): Normalized absolute path to the input file

        Returns:
            list[str]: UUID of the item if we already read it or empty string if we haven't
        """
        return [item_uuid for item_uuid, filepath in self._imported_files.items() if filepath == abs_path]

    def have_read_output_points(self, filename, parameters):
        """Check if we have already imported the same output points csv with the same attributes.

        Args:
            filename (str): Normalized absolute path to the output points csv
            parameters (dict): Dict of the coverage attributes read from the control file for the coverage

        Returns:
            bool: True if we have already read the output points CSV with a simulation that had the same attributes
        """
        cov_uuids = self.check_for_imported_file(filename)
        if not cov_uuids:
            return False
        # We have read the same csv, check if it had the same attributes in its control file.
        for cov_uuid in cov_uuids:
            parameters['cov_uuid'] = cov_uuid  # Don't let the coverage UUID contribute to the compare
            if self._output_points[cov_uuid] == parameters:
                self._logger.info('Duplicate Output Points coverages found in multiple simulations. The same coverage '
                                  'will be used in the simulations.')
                return True
        self._logger.info('Duplicate Output Points coverages found in multiple simulations with different attributes. '
                          'Separate coverages will be created.')
        return False

    def have_read_2dm(self, filename):
        """Check if we have already imported a 2dm file.

        Args:
            filename (str): Normalized absolute path to the 2dm

        Returns:
            str: UUID of the UGrid if we already read it or empty string if we haven't
        """
        uuids = self.check_for_imported_file(filename)
        return uuids[0] if uuids else ''

    def have_read_2dm_bcs(self, filename, bc_parameters):
        """Check if we have already imported a set of BC attributes in 2dm file.

        Don't call this before checking if we have read the file at all with have_read_2dm()

        Args:
            filename (str): Normalized absolute path to the 2dm
            bc_parameters (dict): The BC attributes

        Returns:
            str: UUID of the BC coverage if we already read it or empty string if we haven't
        """
        bc_uuids = self._2dm_bcs.get(filename, [])
        if not bc_uuids or any([bc_uuid is None for bc_uuid in bc_uuids]):
            return None  # None indicates the .2dm file does not contain bcs

        for bc_uuid in bc_uuids:
            check_parameters = self._bcs.get(bc_uuid, {})
            for k, v in check_parameters.items():
                # Add the 'name' attribute of existing BCs since they do not get set until after reading the .2dm.
                # Will get overwritten later if the parameters are different.
                if k not in bc_parameters:
                    return ''  # not equal
                bc_parameters[k]['name'] = v['name']
            if check_parameters == bc_parameters:
                self._logger.info(
                    'Duplicate 2dm BC coverages found in multiple simulations. The same coverage will be used in the '
                    'simulations.'
                )
                return bc_uuid
        if bc_uuids:
            self._logger.info('Duplicate 2dm BC coverages found in multiple simulations with different attributes. '
                              'Separate coverages will be created.')
        return ''

    def have_read_2dm_materials(self, filename, mat_parameters):
        """Check if we have already imported a set of Materials attributes in 2dm file.

        Don't call this before checking if we have read the file at all with have_read_2dm()

        Args:
            filename (str): Normalized absolute path to the 2dm
            mat_parameters (dict): The Materials attributes

        Returns:
            str: UUID of the Materials coverage if we already read it or empty string if we haven't
        """
        mat_uuids = self._2dm_mats.get(filename, [])
        for mat_uuid in mat_uuids:
            if self._materials.get(mat_uuid, {}) == mat_parameters:
                self._logger.info(
                    'Duplicate 2dm Materials coverages found in multiple simulations. The same coverage will be used '
                    'in the simulations.'
                )
                return mat_uuid
        if mat_uuids:
            self._logger.info('Duplicate 2dm Materials coverages found in multiple simulations with different '
                              'attributes. Separate coverages will be created.')
        return ''

    def have_read_gis_materials(self, filename, mat_parameters):
        """Check if we have read a GIS materials shapefile with the same attributes.

        Args:
            filename (str): Normalized absolute path to the shapefile
            mat_parameters (dict): The Materials attributes

        Returns:
            str: UUID of the converted coverage if we already read it or empty string if we haven't
        """
        cov_uuids = self.check_for_imported_file(filename)
        if not cov_uuids:
            return ''
        # We have read the same shapefile, check if it had the same attributes in its control file
        for cov_uuid in cov_uuids:
            if self._materials[cov_uuid] == mat_parameters:
                self._logger.info('Duplicate GIS Materials coverages found in multiple simulations. The same coverage '
                                  'will be used in the simulations.')
                return cov_uuid
        self._logger.info('Duplicate GIS Materials coverages found in multiple simulations with different attributes. '
                          'Separate coverages will be created.')
        return ''

    def have_read_gis_bc(self, filename, bc_parameters):
        """Check if we have read a GIS BC shapefile with the same attributes.

        Args:
            filename (str): Normalized absolute path to the shapefile
            bc_parameters (dict): The BC attributes

        Returns:
            str: UUID of the converted coverage if we already read it or empty string if we haven't
        """
        cov_uuids = self.check_for_imported_file(filename)
        if not cov_uuids:
            return ''
        # We have read the same shapefile, check if it had the same attributes in its control file
        for cov_uuid in cov_uuids:
            check_parameters = self._bcs.get(cov_uuid, {})
            for k, v in check_parameters.items():
                # Add the 'name' attribute of existing BCs since they do not get set until after reading the shapefile.
                # Will get overwritten later if the parameters are different.
                if k not in bc_parameters:
                    identical = False  # not equal
                bc_parameters[k]['name'] = v['name']
            if check_parameters == bc_parameters:
                identical = True
                for bc_id, bc_atts in self._bcs[cov_uuid].items():
                    curve1 = bc_atts.get('curve')
                    curve2 = bc_parameters.get(bc_id, {}).get('curve')
                    curves_mismatch = curve1 is not None and curve2 is not None and not curve1.identical(curve2)
                    if curves_mismatch or (curve1 is None) != (curve2 is None):
                        identical = False
                        break
                if not identical:
                    continue
                self._logger.info('Duplicate GIS BC coverages found in multiple simulations. The same coverage will be '
                                  'used in the simulations.')
                return cov_uuid
        self._logger.info('Duplicate GIS BC coverages found in multiple simulations with different attributes. '
                          'Separate coverages will be created.')
        return ''

    def add_imported_file_no_atts(self, item_uuid, filename):
        """Adds an imported file that has no attributes outside of the file.

        Args:
            item_uuid (str): UUID of the item
            filename (str): Absolute normalized path to the file
        """
        self._imported_files[item_uuid] = filename

    def add_imported_2dm_data(self, filename, ugrid_uuid, bc_uuid, mat_uuid, bc_parameters, mat_parameters):
        """Add imported data from a 2dm.

        Args:
            filename (str): Absolute path to the 2dm file
            ugrid_uuid (str): UUID of the UGrid
            bc_uuid (str): UUID of the BC coverage, empty string if none
            mat_uuid (dict): UUID of the Materials coverage
            bc_parameters (dict): The BC attributes
            mat_parameters (dict): The Materials attributes
        """
        if ugrid_uuid and ugrid_uuid not in self._imported_files:
            self._imported_files[ugrid_uuid] = filename
        if bc_uuid or bc_uuid is None:
            self._2dm_bcs.setdefault(filename, []).append(bc_uuid)
            if bc_uuid:
                self._bcs[bc_uuid] = bc_parameters
        if mat_uuid:
            self._2dm_mats.setdefault(filename, []).append(mat_uuid)
            self._materials[mat_uuid] = mat_parameters

    def add_imported_gis_bc(self, filename, cov_uuid, parameters):
        """Add info about an imported BC coverage converted from a shapefile.

        Args:
            cov_uuid (str): UUID of the coverage
            filename (str): Normalized absolute path to the shapefile
            parameters (dict): Dict of the coverage attributes read from the control file for the coverage
        """
        self._imported_files[cov_uuid] = filename
        self._bcs[cov_uuid] = parameters

    def add_imported_gis_materials(self, filename, cov_uuid, parameters):
        """Add info about an imported Materials coverage converted from a shapefile.

        Args:
            cov_uuid (str): UUID of the coverage
            filename (str): Normalized absolute path to the shapefile
            parameters (dict): Dict of the coverage attributes read from the control file for the coverage
        """
        self._imported_files[cov_uuid] = filename
        self._materials[cov_uuid] = parameters

    def add_imported_output_points(self, filename, cov_uuid, parameters):
        """Add info about an imported Output Points coverage.

        Args:
            cov_uuid (str): UUID of the coverage
            filename (str): Normalized absolute path to the output points csv
            parameters (dict): Dict of the coverage attributes read from the control file for the coverage
        """
        self._imported_files[cov_uuid] = filename
        self._output_points[cov_uuid] = parameters

    def get_gis_zline_uuid(self, filename):
        """Gets the UUID of a previously read point or line Z shapefile or generates a new one.

        Args:
            filename (str): Normalized absolute path to the shapefile

        Returns:
            str: UUID of the previously read shapefile, or a randomized UUID if we have not
        """
        uuids = self.check_for_imported_file(filename)  # Should only ever have 0 or 1 UUID
        if not uuids:
            uuids = [str(uuid.uuid4())]
            self._imported_files[uuids[0]] = filename
        return uuids[0]

    def get_gis_po_uuid(self, filename):
        """Gets the UUID of a previously read output point or line Z shapefile or generates a new one.

        Args:
            filename (str): Normalized absolute path to the shapefile

        Returns:
            str: UUID of the previously read shapefile, or empty string if not found
        """
        cov_uuid = ''
        uuids = self.check_for_imported_file(filename)
        if uuids:  # Should only ever have 0 or 1 UUID
            cov_uuid = uuids[0]
        return cov_uuid
