"""Class to read TUFLOWFV Holland Wind boundaries."""
# 1. Standard python modules
import datetime
import logging
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.coverage.windCoverage import WindCoverage, WindModel_enum, WindNode
from xms.data_objects.parameters import Arc, Coverage, datetime_to_julian, Point

# 4. Local modules
from xms.tuflowfv.file_io.bc_csv_reader import BcCsvReader


class HollandWindReader:
    """Class to read a TUFLOWFV Holland Wind boundaries."""
    COLUMN_HEADER = [
        'Time', 'X', 'Y', 'P0', 'PA', 'RMAX', 'B', 'RHOA', 'KM', 'THETMAX', 'DELTAFM', 'Latitude', 'WBGX', 'WBGY',
    ]
    METERS_PER_NAUTICAL_MILE = 1852

    def __init__(self, reftime, do_projection):
        """Constructor.

        Args:
            reftime (datetime.datetime): The simulation reference date
            do_projection (Projection): Projection of the simulation's geometric data
        """
        if not reftime:
            reftime = datetime.datetime(1990,1,1)
        self._reftime = reftime
        self._do_projection = do_projection
        self._logger = logging.getLogger('xms.tuflowfv')

    def _build_geometry(self, df, cov_name):
        """Create feature points and arcs from the locations in the CSV.

        Args:
            df (DataFrame): The DataFrame as read from the input CSV file
            cov_name (str): Tree item name to give the Coverage

        Returns:
            data_objects.parameters.Coverage: The coverage geometry
        """
        x_locs = df['X'].values.tolist()
        y_locs = df['Y'].values.tolist()
        cov_pts = [Point(x=x_loc, y=y_loc, feature_id=i + 1) for i, (x_loc, y_loc) in enumerate(zip(x_locs, y_locs))]
        cov_arcs = [Arc(feature_id=i, start_node=cov_pts[i - 1], end_node=cov_pts[i]) for i in range(1, len(cov_pts))]
        do_cov = Coverage(name=cov_name, uuid=str(uuid.uuid4()), projection=self._do_projection)
        do_cov.arcs = cov_arcs
        do_cov.complete()
        return do_cov

    def _build_wind_dump(self, df, uses_isodate):
        """Build a WindCoverage dump from data imported from the CSV.

        Args:
            df (DataFrame): The imported CSV data
            uses_isodate (bool): True if time column is isodate, False if it is hour offsets

        Returns:
            WindCoverage: The built wind coverage dump
        """
        wind_dump = WindCoverage()
        wind_dump.m_windModel = WindModel_enum.WM_HOLLAND_SIMPLIFIED_ASYMMETRIC
        for index, row in df.iterrows():
            wind_node = WindNode()
            wind_node.m_id = index  # CSV reader makes index 1-based row number for GUI
            dt = row['Time'].to_pydatetime() if uses_isodate else self._reftime + datetime.timedelta(hours=row['Time'])
            wind_node.m_date = datetime_to_julian(dt)
            wind_node.m_iMinSeaLevelPressure = int(row['P0'])
            wind_node.m_iPressure = int(row['PA'])
            wind_node.m_iMaxWindRadius = int(round(row['RMAX'] / self.METERS_PER_NAUTICAL_MILE))
            wind_node.m_hollandB = float(row['B'])
            wind_node.rhoa = float(row['RHOA'])
            wind_node.km = float(row['KM'])
            wind_node.m_iStormDirection = int(row['THETMAX'])
            wind_node.deltafm = int(row['DELTAFM'])
            wind_node.wbgx = float(row['WBGX'])
            wind_node.wbgy = float(row['WBGY'])
            wind_dump.m_nodeWind.append(wind_node)
        return wind_dump

    def read(self, filename):
        """Read a TUFLOWFV Holland wind boundary CSV file.

        Args:
            filename (str): The path to the CSV file to read

        Returns:
            WindCoverage: The imported wind coverage dump
        """
        try:
            self._logger.info('Importing Holland wind boundary from CSV file...')
            # , filename, default_columns, user_columns, x_is_time):
            csv_reader = BcCsvReader(filename=filename, default_columns=self.COLUMN_HEADER,
                                     user_columns=self.COLUMN_HEADER, x_is_time=True)
            df = csv_reader.read()
            do_cov = self._build_geometry(df=df, cov_name=os.path.splitext(os.path.basename(filename))[0])
            wind_dump = self._build_wind_dump(df=df, uses_isodate=csv_reader.uses_isodate)
            wind_dump.m_cov = do_cov
            return wind_dump
        except Exception as e:
            self._logger.error(f'Errors reading CSV file: {str(e)}')
        return None
