"""Creates an output points coverage hidden component and Datasets."""
# 1. Standard python modules
import logging
import os
import uuid

# 2. Third party modules
import xarray as xr

# 3. Aquaveo modules
from xms.components.display.display_options_io import write_display_option_ids

# 4. Local modules
from xms.tuflowfv.components import output_points_component as opc
from xms.tuflowfv.components import output_points_component_display as opc_disp
from xms.tuflowfv.components.tuflowfv_component import get_component_data_object
from xms.tuflowfv.data import output_points_data as opd
from xms.tuflowfv.file_io.io_util import create_component_folder


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


def build_output_dataset(att_dict, comp_id=opd.UNINITIALIZED_COMP_ID):
    """Build an xarray Dataset for a single output point and fill with values read from the .fvc file.

    Args:
        att_dict (dict): The variables for the output point read from the .fvc file.
        comp_id (int): Comp id to assign to the new dataset. Leave default if appending the output point to the
            OutputPointsData as that will increment the component id to the next available one.

    Returns:
        xr.Dataset: See description
    """
    default_data = opd.get_default_output_data(fill=True)
    coords = {'comp_id': [comp_id]}  # This will get reset when we append to the entire Dataset
    dset = xr.Dataset(data_vars=default_data, coords=coords)
    for variable, value in att_dict.items():
        dset[variable].loc[dict(comp_id=[comp_id])] = value
    return dset


class OutputComponentBuilder:
    """Class for building output points components."""

    def __init__(self, cov_uuid, from_csv, atts, point_id_to_feature_id, point_id_to_feature_name):
        """Constructor.

        Args:
            cov_uuid (str): UUID of the component's coverage geometry
            from_csv (bool): True if coverage is read from the .CSV file or the points-only coverage in the .fvc file.
                If False (read from a GIS shapefile), we will export as a shapefile when exporting the simulation.
            atts (dict): {point_id: {variable: value}} The point attributes read from the CSV
            point_id_to_feature_id (dict): Mapping of point id in the file to feature arc id we created
            point_id_to_feature_name (dict): Mapping of point id in the file to feature arc name from the
                file. Since the feature names are optional, there may not be a value for a given key
        """
        self._logger = logging.getLogger('xms.tuflowfv')
        self._cov_uuid = cov_uuid
        self._from_csv = from_csv
        self._atts = atts
        self._point_id_to_feature_id = point_id_to_feature_id
        self._point_id_to_feature_name = point_id_to_feature_name
        self._main_file = ''

    def _initialize_component(self):
        """Create the component's folder and data_object.

        Returns:
            xms.data_objects.parameters.Component: data_object for the new BC component
        """
        # Create a new UUID and folder for the component data
        comp_uuid = str(uuid.uuid4())
        comp_dir = create_component_folder(comp_uuid)
        self._main_file = os.path.join(comp_dir, opd.OUTPUT_POINTS_MAIN_FILE)
        # Create the data_object Component to send back to SMS
        return get_component_data_object(main_file=self._main_file, comp_uuid=comp_uuid,
                                         unique_name='OutputPointsComponent')

    def _initialize_display(self, py_comp, att_ids, comp_ids):
        """Initialize the component display.

        Args:
            py_comp (OutputPointsComponent): The Python BC component object
            att_ids (list[int]): The point feature ids
            comp_ids (list[int]): The point component ids
        """
        display_helper = opc_disp.OutputPointsComponentDisplay(py_comp)
        display_helper.update_id_files()
        # Write component id and att ids to a file so we can initialize them in get_initial_display_options
        comp_dir = os.path.dirname(self._main_file)
        id_file = os.path.join(comp_dir, opc_disp.OUTPUT_INITIAL_ATT_ID_FILE)
        write_display_option_ids(id_file, att_ids)
        id_file = os.path.join(comp_dir, opc_disp.OUTPUT_INITIAL_COMP_ID_FILE)
        write_display_option_ids(id_file, comp_ids)

    def _add_atts(self, data):
        """Add attribute data for the features.

        Args:
            data (OutputPointsData): The OutputPointsData data

        Returns:
            tuple: point att ids, point comp ids
        """
        att_ids = []
        comp_ids = []
        for point_id, feature_id in self._point_id_to_feature_id.items():
            # Append a row to the Dataset for this BC
            att_dict = self._atts.get(point_id)
            if not att_dict and self._point_id_to_feature_name:  # Check if it had a string name in the file
                point_id = self._point_id_to_feature_name.get(point_id)
                att_dict = self._atts.get(point_id)
            if not att_dict:
                continue
            comp_id = data.add_output_points(dset=build_output_dataset(att_dict))
            # Keep track of the component and feature ids for initializing display later.
            att_ids.append(feature_id)
            comp_ids.append(comp_id)
        return att_ids, comp_ids

    def build_output_component(self):
        """Creates the output points coverage's hidden component.

        This method is for building a output points coverage from scratch.

        Returns:
            xms.data_objects.parameters.Component: data_object for the new output points component
        """
        # Set things up
        do_comp = self._initialize_component()
        py_comp = opc.OutputPointsComponent(self._main_file)
        comp_data = py_comp.data
        comp_data.info.attrs['cov_uuid'] = self._cov_uuid
        comp_data.info.attrs['export_format'] = 'CSV' if self._from_csv else 'Shapefile'
        py_comp.cov_uuid = comp_data.info.attrs['cov_uuid']

        # Add the output points
        att_ids, comp_ids = self._add_atts(comp_data)
        # Write data to disk
        comp_data.commit()
        self._initialize_display(py_comp, att_ids, comp_ids)
        return do_comp
