"""Widget to display line numbers and current line highlighting in a text widget."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import QRect, QRegExp, Qt
from PySide2.QtGui import QColor, QFont, QPainter, QSyntaxHighlighter, QTextCharFormat, QTextFormat
from PySide2.QtWidgets import QPlainTextEdit, QTextEdit, QWidget

# 3. Aquaveo modules

# 4. Local modules


"""
Python3 and PySide2 adaption of this solution: https://github.com/luchko/QCodeEditor
"""


class TuflowFVHighlighter(QSyntaxHighlighter):
    """Class to provide basic syntax highlighting for TUFLOWFV .fvc files."""

    def __init__(self, parent=None):
        """Constructor.

        Args:
            parent (QtWidget): The parent widget
        """
        super().__init__(parent)
        self.highlighting_rules = []

        command_format = QTextCharFormat()
        command_format.setForeground(QColor("#000070"))  # blue
        self.highlighting_rules.append((QRegExp(r'\b[A-Za-z0-9_-\s]+(?= \=)'), command_format))
        self.highlighting_rules.append((QRegExp('end.*'), command_format))

        command_operator_format = QTextCharFormat()
        command_operator_format.setFontItalic(True)
        command_operator_format.setForeground(QColor(Qt.red))  # green
        self.highlighting_rules.append((QRegExp('='), command_operator_format))

        comment_format = QTextCharFormat()
        comment_format.setForeground(QColor(Qt.darkGreen))
        self.highlighting_rules.append((QRegExp('![^\n]*'), comment_format))

    def highlightBlock(self, text):  # noqa: N802
        """Override highlight block for custom rules."""
        # for every pattern
        for pattern, rule_format in self.highlighting_rules:
            # Create a regular expression from the retrieved pattern
            expression = QRegExp(pattern)
            # Check what index that expression occurs at with the ENTIRE text
            index = expression.indexIn(text)
            # While the index is greater than 0
            while index >= 0:
                # Get the length of how long the expression is true, set the format from the start to the length with
                # the text format
                length = expression.matchedLength()
                self.setFormat(index, length, rule_format)
                # Set index to where the expression ends in the text
                index = expression.indexIn(text, index + length)


class NumberBar(QWidget):
    """Class that defines textEditor number bar."""

    def __init__(self, code_editor):
        """Constructor.

        Args:
            code_editor (CodeEditorWidget): The parent code editor
        """
        super().__init__(code_editor)
        self.editor = code_editor
        self.editor.blockCountChanged.connect(self.update_width)
        self.editor.updateRequest.connect(self.update_contents)
        self.font = QFont()
        self.number_bar_color = QColor('#e8e8e8')

    def paintEvent(self, event):  # noqa: N802
        """Overload paintEvent handler."""
        painter = QPainter(self)
        painter.fillRect(event.rect(), self.number_bar_color)
        block = self.editor.firstVisibleBlock()
        # Iterate over all visible text blocks in the document.
        while block.isValid():
            block_number = block.blockNumber()
            block_top = self.editor.blockBoundingGeometry(block).translated(self.editor.contentOffset()).top()
            # Check if the position of the block is out side of the visible area.
            if not block.isVisible() or block_top >= event.rect().bottom():
                break
            # We want the line number for the selected line to be bold.
            if block_number == self.editor.textCursor().blockNumber():
                self.font.setBold(True)
                painter.setPen(QColor('#000000'))
            else:
                self.font.setBold(False)
                painter.setPen(QColor('#717171'))
            painter.setFont(self.font)
            # Draw the line number right justified at the position of the line.
            paint_rect = QRect(0, block_top, self.width(), self.editor.fontMetrics().height())
            painter.drawText(paint_rect, Qt.AlignRight, str(block_number + 1))
            block = block.next()  # noqa: B305   - Not a Python2 next(), QTextBlock.next()
        painter.end()
        QWidget.paintEvent(self, event)

    def get_width(self):
        """Get the width of the editor."""
        count = self.editor.blockCount()
        width = self.fontMetrics().width(str(count)) + 10
        return width

    def update_width(self):
        """Update the width of the editor."""
        width = self.get_width()
        if self.width() != width:
            self.setFixedWidth(width)
            self.editor.setViewportMargins(width, 0, 0, 0)

    def update_contents(self, rect, scroll):
        """Update the contents of the editor."""
        if scroll:
            self.scroll(0, scroll)
        else:
            self.update(0, rect.y(), self.width(), rect.height())
        if rect.contains(self.editor.viewport().rect()):
            font_size = self.editor.currentCharFormat().font().pointSize()
            self.font.setPointSize(font_size)
            self.font.setStyle(QFont.StyleNormal)
            self.update_width()


class CodeEditorWidget(QPlainTextEdit):
    """QCodeEditor inherited from QPlainTextEdit providing."""

    def __init__(self, syntax_highlighter_class):
        """Constructor.

        Args:
            syntax_highlighter_class (class): Class (not instance) of syntax highlighter to use. Must derive from
                QSyntaxHighlighter
        """
        super().__init__()
        self.setFont(QFont("Ubuntu Mono", 11))
        self.setLineWrapMode(QPlainTextEdit.NoWrap)
        self.number_bar = NumberBar(self)
        self.current_line_no = None
        self.current_line_color = self.palette().alternateBase()
        self.cursorPositionChanged.connect(self.highlight_current_line)
        if syntax_highlighter_class is not None:  # add highlighter to textdocument
            self.highlighter = syntax_highlighter_class(self.document())
        else:
            self.highlighter = None
        self.setViewportMargins(self.number_bar.get_width(), 0, 0, 0)

    def resizeEvent(self, *e):  # noqa: N802
        """Overload resizeEvent handler."""
        cr = self.contentsRect()
        rec = QRect(cr.left(), cr.top(), self.number_bar.get_width(), cr.height())
        self.number_bar.setGeometry(rec)
        QPlainTextEdit.resizeEvent(self, *e)

    def highlight_current_line(self):
        """Slot called when current position changes."""
        current_line_no = self.textCursor().blockNumber()
        if current_line_no != self.current_line_no:
            self.current_line_no = current_line_no
            hi_selection = QTextEdit.ExtraSelection()
            hi_selection.format.setBackground(self.current_line_color)
            hi_selection.format.setProperty(QTextFormat.FullWidthSelection, True)
            hi_selection.cursor = self.textCursor()
            hi_selection.cursor.clearSelection()
            self.setExtraSelections([hi_selection])
