"""Qt delegate that displays a button for bringing up the Assign BC dialog."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, Qt
from PySide2.QtGui import QBrush, QColor, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.gui.assign_bc_dialog import AssignBcDialog


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


def run_global_bc_dialog(bc_data, comp_id, parent, bc_location, time_formats):
    """Run the Assign BC dialog for a global/gridded BC.

    Args:
        bc_data (BcData): The global BCs Dataset
        comp_id (int): Comp id of the row to edit
        parent (QWidget): The Qt parent dialog
        bc_location (int): Location of the BC, one of BC_LOCATION_* constants
        time_formats (tuple(str)): The current SMS user preference for formatting absolute datetimes. Should have
            specifiers for (strftime, qt)
    """
    bc_row = bc_data.bcs.where(bc_data.bcs.comp_id == comp_id, drop=True)
    bc_type = bc_row.type.item()
    # Rename the curve columns for the GUI if QG type. Keep the original as a copy in case user cancels. TUFLOWFV does
    # not like the underscores, NetCDF can't handle path separators.
    orig_bc_curve = None
    if bc_type == 'QG':
        bc_curve = bc_data.get_bc_curve(comp_id=comp_id, bc_type=bc_type, default=True)
        orig_bc_curve = bc_curve.copy(deep=True)
        bc_curve = orig_bc_curve.rename({'Q_over_A': 'Q/A'})
        bc_data.set_bc_curve(comp_id=comp_id, bc_type=bc_type, dataset=bc_curve)
    dlg = AssignBcDialog(parent=parent, bc_data=bc_row, all_data=bc_data, msg='', bc_location=bc_location,
                         comp_id=comp_id, time_formats=time_formats)
    if dlg.exec():
        bc_data.update_bc(comp_id=comp_id, new_atts=dlg.bc_data, bc_location=bc_location)
        if bc_type == 'QG':  # Make curve columns NetCDF compatible again
            curve = bc_data.get_bc_curve(comp_id=comp_id, bc_type=bc_type, default=True)
            curve = curve.rename({'Q/A': 'Q_over_A'})
            bc_data.set_bc_curve(comp_id=comp_id, bc_type=bc_type, dataset=curve)
    elif orig_bc_curve is not None:  # Restore the original curve if user cancels
        bc_data.set_bc_curve(comp_id=comp_id, bc_type=bc_type, dataset=orig_bc_curve)


class DefineBcButtonDelegate(QStyledItemDelegate):
    """Qt delegate that displays a button for bringing up the Assign BC dialog in the model control dialog."""

    def __init__(self, parent, bc_data, bc_location, time_formats):
        """Initializes the class.

        Args:
            parent (QObject): The parent object.
            bc_data (BcData): The BcData to add/edit BCs
            bc_location (int): Location of the BC, one of BC_LOCATION_* constants
            time_formats (tuple(str)): The current SMS user preference for formatting absolute datetimes. Should have
                specifiers for (strftime, qt)
        """
        super().__init__(parent)
        self._parent_dlg = parent
        self._bc_data = bc_data
        self._bc_location = bc_location
        self._time_formats = time_formats

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            btn.setText('Assign BC...')
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())
            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)
        else:
            painter.fillRect(option.rect, QColor(240, 240, 240))

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.

        Returns:
            bool: True if the event is handled
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                row = index.row()
                comp_id = self._bc_data.bcs.coords['comp_id'][row].item()
                run_global_bc_dialog(bc_data=self._bc_data, comp_id=comp_id, parent=self._parent_dlg,
                                     bc_location=self._bc_location, time_formats=self._time_formats)
                return True
        return super().editorEvent(event, model, option, index)
