"""Filter model for the material properties table."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import QSortFilterProxyModel, Qt

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.data.material_data import MaterialData

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


COLUMN_TEXT = [  # Pretty text with newlines for the GUI.
    'ID',  # Should be hidden, but keep parallel with 'COL_*' constants in MaterialData
    'Color and Texture',
    'Name',
    'Inactive',
    'Override\nBottom Roughness',
    'Bottom Roughness',
    'Advanced options',
]


class MaterialFilterModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)

    def _set_flags_from_toggle(self, model_idx, row_idx, col_idx, flags):
        """Set window flags based on toggle state.

        Args:
            model_idx (QModelIndex): The item's Qt model index
            row_idx (int): Index of the item row
            col_idx (int): Index of the toggle column
            flags (Qt.ItemFlags): The flags for the item

        Returns:
            Qt.ItemFlags: The updated flags for the item
        """
        checked = model_idx.model().index(row_idx, col_idx).data(Qt.CheckStateRole)
        if checked == Qt.Checked:
            flags |= Qt.ItemIsEnabled
            flags |= Qt.ItemIsEditable
        else:
            flags = flags & (~Qt.ItemIsEnabled)
            flags = flags & (~Qt.ItemIsEditable)
        return flags

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (int): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (int): The role.

        Returns:
            The data.
        """
        if role != Qt.DisplayRole:
            return super().headerData(section, orientation, role)

        if orientation == Qt.Horizontal:
            if section >= len(COLUMN_TEXT):
                return ''  # Watch out for the hidden columns
            return COLUMN_TEXT[section]
        else:
            return super().headerData(section, orientation, role)

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (QModelIndex): The item's model index

        Returns:
            Qt.ItemFlags: Updated flags for the item
        """
        flags = super().flags(index)
        row = index.row()
        col = index.column()

        if row == MaterialData.UNASSIGNED_MAT:
            if col == MaterialData.COL_NAME:  # Disable editing of the unassigned material name
                flags = flags & (~Qt.ItemIsEditable)
            elif col != MaterialData.COL_COLOR:
                flags = flags & (~Qt.ItemIsEnabled)
            else:
                flags |= Qt.ItemIsEnabled
        elif col == MaterialData.COL_BOTTOM_ROUGHNESS:
            flags = self._set_flags_from_toggle(index, row, MaterialData.COL_OVERRIDE_BOTTOM_ROUGHNESS, flags)
        elif col == MaterialData.COL_ADVANCED:
            flags = flags & (~Qt.ItemIsEditable)
            flags |= Qt.ItemIsEnabled
        else:
            flags |= Qt.ItemIsEnabled
        return flags
