"""Parent dialog for the Materials and Sediment Materials properties dialogs."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QLabel, QVBoxLayout

# 3. Aquaveo modules
from xms.core.filesystem import filesystem
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.tuflowfv.data.material_data import MaterialData
from xms.tuflowfv.gui.material_dialog import MaterialDialog

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


class TfMaterialParentDialog(XmsDlg):
    """Container for the material dialog. Makes importing data very simple."""

    def __init__(self, title, parent, material_data, display_projection, selected_material=None):
        """Initializes the material list and properties dialog.

        Args:
            title (str): Title of the dialog
            parent (QWidget): Parent dialog
            material_data (MaterialData): The material data to edit
            display_projection (Projection): The current display projection. Used for switching unit labels between SI
                and Imperial.
            selected_material (int): 0-based row index of the material that should be selected. If provided, implies
                the dialog is coming from the Assign Material command.
        """
        super().__init__(parent, 'xms.tuflowfv.gui.material_parent_dialog')
        self.title = title
        self.material_dialog = None
        self.selected_material = selected_material
        self.display_projection = display_projection
        self.widgets = dict()
        self.widgets['main_vert_layout'] = QVBoxLayout()
        self.setLayout(self.widgets['main_vert_layout'])
        self.import_file(material_data)
        self.setWindowTitle(title)

    def add_multi_polygon_select_warning(self):
        """Add a label if coming from the Assign Material command with multiple polygons selected."""
        self.material_dialog.widgets['lbl_multi_select'] = QLabel(
            'Material assignment will be applied to all selected polygons.'
        )
        self.material_dialog.widgets['lbl_multi_select'].setStyleSheet('font-weight: bold; color: red')
        self.material_dialog.widgets['main_vert_layout'].insertWidget(
            0, self.material_dialog.widgets['lbl_multi_select']
        )

    def accept(self):
        """Accept the dialog."""
        super().accept()

    def reject(self):
        """Reject the dialog."""
        super().reject()

    def import_file(self, material_data):
        """Loads a material properties file.

        Args:
            material_data (MaterialData): The material data to edit
        """
        if self.material_dialog is not None:  # Overwrite the old data with the newly imported one.
            old_filename = self.material_dialog.material_data._filename
            attrs = self.material_dialog.material_data.info.attrs
            self.material_dialog.material_data.close()
            filesystem.copyfile(material_data._filename, old_filename)
            material_data = MaterialData(old_filename)
            material_data.info.attrs = attrs
            material_data.commit()
        else:
            material_data = material_data

        # if the layout is not empty then remove the dialog from it
        child = self.widgets['main_vert_layout'].takeAt(0)
        if child is not None:
            if child.widget() is not None:
                child.widget().deleteLater()

        dlg = MaterialDialog(title=self.title, parent=self, material_data=material_data,
                             display_projection=self.display_projection)
        dlg.setWindowFlags(dlg.windowFlags() & ~Qt.Dialog)
        self.material_dialog = dlg
        self.widgets['main_vert_layout'].addWidget(self.material_dialog)
