"""Qt delegate that displays a button for bringing up the Z point dialog in the model control dialog."""
# 1. Standard python modules

# 2. Third party modules
import numpy as np
from PySide2.QtCore import QEvent, Qt
from PySide2.QtGui import QBrush, QColor, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.data import sim_data as smd
from xms.tuflowfv.gui import gui_util
from xms.tuflowfv.gui.tree_item_selector_table_dialog import TreeItemSelectorTableDialog


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


MAX_POINT_SHAPEFILES = 9
NON_ZPOINT_VARS = [
    smd.ELEV_MOD_TYPE_VAR,
    smd.ELEV_SET_ZPTS_VAR,
    smd.ELEV_GRID_ZPTS_VAR,
    smd.ELEV_CSV_FILE_VAR,
    smd.ELEV_CSV_TYPE_VAR,
    smd.ELEV_ZLINE_UUID_VAR,
]


class ZPointsButtonDelegate(QStyledItemDelegate):
    """Qt delegate that displays a button for bringing up the Z point dialog in the model control dialog."""

    def __init__(self, parent, tree_node):
        """Initializes the class.

        Args:
            parent (QObject): The parent object.
            tree_node (TreeNode): The root project explorer tree
        """
        super().__init__(parent)
        self._parent_dlg = parent
        self._tree_node = tree_node

    def _update_dataframe(self, row, new_values):
        """Update a row in the main dataframe after a user changed the z point selections.

        Args:
            row (int): Index of the row to update
            new_values (pd.DataFrame): The new values for the row
        """
        df = self._parent_dlg.z_modifications_table.model.data_frame
        point_uuids = new_values[new_values.columns[0]].values
        point_uuids = point_uuids[~np.isin(point_uuids, ['', gui_util.NULL_SELECTION])]  # Filter out invalid selections
        num_layers = len(point_uuids)
        df[smd.ELEV_ZPOINT1_UUID_VAR].values[row] = point_uuids[0] if num_layers > 0 else ''
        df[smd.ELEV_ZPOINT2_UUID_VAR].values[row] = point_uuids[1] if num_layers > 1 else ''
        df[smd.ELEV_ZPOINT3_UUID_VAR].values[row] = point_uuids[2] if num_layers > 2 else ''
        df[smd.ELEV_ZPOINT4_UUID_VAR].values[row] = point_uuids[3] if num_layers > 3 else ''
        df[smd.ELEV_ZPOINT5_UUID_VAR].values[row] = point_uuids[4] if num_layers > 4 else ''
        df[smd.ELEV_ZPOINT6_UUID_VAR].values[row] = point_uuids[5] if num_layers > 5 else ''
        df[smd.ELEV_ZPOINT7_UUID_VAR].values[row] = point_uuids[6] if num_layers > 6 else ''
        df[smd.ELEV_ZPOINT8_UUID_VAR].values[row] = point_uuids[7] if num_layers > 7 else ''
        df[smd.ELEV_ZPOINT9_UUID_VAR].values[row] = point_uuids[8] if num_layers > 8 else ''

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            btn.setText('Select Z Point Layers...')
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())
            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)
        else:
            painter.fillRect(option.rect, QColor(240, 240, 240))

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.

        Returns:
            bool: True if the event is handled
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                row = index.row()
                # Select the current row from the DataFrame, drop the line shape column, and transpose so point layers
                # are now rows with a single column for UUID.
                df = self._parent_dlg.z_modifications_table.model.data_frame
                df = df.iloc[row].drop(NON_ZPOINT_VARS).transpose().to_frame()
                df = df[df[df.columns[0]] != '']  # Drop rows that have not been defined.
                df.index = [i + 1 for i in range(len(df))]
                dlg = TreeItemSelectorTableDialog(title='Select Z Point Layers', tree_node=self._tree_node,
                                                  tree_type='TI_GENERIC_POINT', layer_type='Z', data_frame=df,
                                                  max_rows=MAX_POINT_SHAPEFILES, parent=self.parent())
                if dlg.exec():
                    self._update_dataframe(row, dlg.selector_table.model.data_frame)
                return True
        return super().editorEvent(event, model, option, index)
