"""Module for the get_model function."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['get_output_field_model']

# 1. Standard Python modules
import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules


def get_output_field_model() -> GenericModel:
    """Create the hard bottom data definition."""
    gm = GenericModel()

    # Time parameters
    time_parameters = gm.model_parameters.add_group('time_parameters', 'Time parameters')

    default = datetime.datetime(1950, 1, 1, 0, 0, 0)
    time_parameters.add_date_time('output_start_date', 'Start Date', default)
    time_parameters.add_integer('output_second_interval', 'Interval Seconds', default=1800)
    time_parameters.add_date_time('output_end_date', 'End Date', default)

    # Forcing fields
    forcing_fields = gm.model_parameters.add_group('forcing_fields', 'Forcing fields parameters')

    forcing_fields.add_boolean('DPT', 'Water depth [DPT]', default=False)
    forcing_fields.add_boolean('CUR', 'Current velocity [CUR]', default=False)
    forcing_fields.add_boolean('WND', 'Wind speed [WND]', default=False)
    forcing_fields.add_boolean('AST', 'Air-sea temperature difference [AST]', default=False)
    forcing_fields.add_boolean('WLV', 'Water levels [WLV]', default=False)
    forcing_fields.add_boolean('ICE', 'Ice concentration [ICE]', default=False)
    forcing_fields.add_boolean('IBG', 'Iceberg-induced damping [IBG]', default=False)
    forcing_fields.add_boolean('D50', 'Median sediment grain size [D50]', default=False)
    forcing_fields.add_boolean('IC1', 'Ice thickness [IC1]', default=False)
    forcing_fields.add_boolean('IC5', 'Ice flow diameter [IC5]', default=False)

    # Standard mean wave parameters
    mean_wave = gm.model_parameters.add_group('mean_wave', 'Standard mean wave parameters')

    mean_wave.add_boolean('HS', 'Wave height [HS]', default=False)
    mean_wave.add_boolean('LM', 'Mean wave length [LM]', default=False)
    mean_wave.add_boolean('T02', 'Mean wave period (Tm0,2) [T02]', default=False)
    mean_wave.add_boolean('T0M1', 'Mean wave period (TM0,-1) [T0M1]', default=False)
    mean_wave.add_boolean('T01', 'Mean wave period (Tm0,1) [T01]', default=False)
    mean_wave.add_boolean('FP', 'Peak frequency [FP]', default=False)
    mean_wave.add_boolean('DIR', 'Mean wave direction [DIR]', default=False)
    mean_wave.add_boolean('SPR', 'Mean directional spread [SPR]', default=False)
    mean_wave.add_boolean('DP', 'Peak direction [DP]', default=False)
    mean_wave.add_boolean('HIG', 'Infragravity height [HIG]', default=False)
    mean_wave.add_boolean('MXE', 'Max surface elev (STE) [MXE]', default=False)
    mean_wave.add_boolean('MXES', 'St Dev of max surface elev (STE) [MXES]', default=False)
    mean_wave.add_boolean('MXH', 'Max wave height (STE) [MXH]', default=False)
    mean_wave.add_boolean('MXHC', 'Max wave height from crest (STE) [MXHC]', default=False)
    mean_wave.add_boolean('SDMH', 'Std Dev of MXC (STE) [SDMH]', default=False)
    mean_wave.add_boolean('SDMHC', 'St Dev of MXHC (STE) [SDMHC]', default=False)
    mean_wave.add_boolean('WBT', 'Dominant wave breaking probability b [WBT]', default=False)
    mean_wave.add_boolean('TP', 'Peak period (from peak freq) [TP]', default=False)

    # Spectral parameters (advanced)
    spectral_parameters = gm.model_parameters.add_group('spectral_parameters', 'Spectral parameters (advanced)')

    spectral_parameters.add_boolean('EF', 'Wave frequency spectrum [EF]', default=False)
    spectral_parameters.add_boolean('TH1M', 'Mean wave direction from a1,b2 [TH1M]', default=False)
    spectral_parameters.add_boolean('STH1M', 'Directional spreading from a1,b2 [STH1M]', default=False)
    spectral_parameters.add_boolean('TH2M', 'Mean wave direction from a2,b2 [TH2M]', default=False)
    spectral_parameters.add_boolean('STH2M', 'Directional spreading from a2,b2 [STH2M]', default=False)
    spectral_parameters.add_boolean('WN', 'Wavenumber array [WN]', default=False)

    # Spectral partition parameters (advanced)
    partition_parameters = gm.model_parameters.add_group('partition_params', 'Spectral partition parameters (advanced)')

    partition_parameters.add_boolean('PHS', 'Partitioned wave heights [PHS]', default=False)
    partition_parameters.add_boolean('PTP', 'Partitioned peak period [PTP]', default=False)
    partition_parameters.add_boolean('PLP', 'Partitioned peak wave length [PLP]', default=False)
    partition_parameters.add_boolean('PDIR', 'Partitioned mean direction [PIDR]', default=False)
    partition_parameters.add_boolean('PSPR', 'Partitioned mean direction spread [PSPR]', default=False)
    partition_parameters.add_boolean('PWS', 'Partitioned wind sea fraction [PWS]', default=False)
    partition_parameters.add_boolean('PDP', 'Peak wave direction of partition [PDP]', default=False)
    partition_parameters.add_boolean('PQP', 'Goda peadedness parameter of partition [PQP]', default=False)
    partition_parameters.add_boolean('PPE', 'JONSWAP peak enchancement factor of partition [PPE]', default=False)
    partition_parameters.add_boolean('PGW', 'Guassian frequency width of partition [PGW]', default=False)
    partition_parameters.add_boolean('PSW', 'Spectral width of partition [PSW]', default=False)
    partition_parameters.add_boolean('PTM10', 'Mean wave period (m-1,0) [PTM10]', default=False)
    partition_parameters.add_boolean('PT01', 'Mean wave period (m0,1) of partition [PT01]', default=False)
    partition_parameters.add_boolean('PT02', 'Mean wave period (m0,2) of partition [PT02]', default=False)
    partition_parameters.add_boolean('PEP', 'Peak spectral density of partition [PEP]', default=False)
    partition_parameters.add_boolean('TWS', 'Total wind sea fraction [TWS]', default=False)
    partition_parameters.add_boolean('PNR', 'Number of partitions [PNR]', default=False)

    # Atmosphere-waves layer (advanced)
    atmosphere_waves = gm.model_parameters.add_group('atmosphere_waves', 'Atmosphere-waves layer (advanced)')

    atmosphere_waves.add_boolean('UST', 'Friction velocity [UST]', default=False)
    atmosphere_waves.add_boolean('CHA', 'Charnock parameters [CHA]', default=False)
    atmosphere_waves.add_boolean('CGE', 'Energy flux [CGE]', default=False)
    atmosphere_waves.add_boolean('FAW', 'Air-sea energy flux [FAW]', default=False)
    atmosphere_waves.add_boolean('TAW', 'Net wave-supported stress [TAW]', default=False)
    atmosphere_waves.add_boolean('TWA', ' Negative part of the wave-supported stress [TWA]', default=False)
    atmosphere_waves.add_boolean('WCC', 'Whitecap coverage [WCC]', default=False)
    atmosphere_waves.add_boolean('WCF', 'Whitecap thickness [WCF]', default=False)
    atmosphere_waves.add_boolean('WCH', 'Mean breaking height [WCH]', default=False)
    atmosphere_waves.add_boolean('WCM', 'Whitecap moment [WCM]', default=False)
    atmosphere_waves.add_boolean('FWS', 'Wind sea mean period [FWS]', default=False)

    # Wave-ocean layer (advanced)
    wave_ocean = gm.model_parameters.add_group('wave_ocean', 'Wave-ocean layer (advanced)')

    wave_ocean.add_boolean('SXY', 'Radiation stresses [SXY]', default=False)
    wave_ocean.add_boolean('TWO', 'Wave to ocean momentum [TWO]', default=False)
    wave_ocean.add_boolean('BHD', 'Bernoulli head (J term) [BHJD]', default=False)
    wave_ocean.add_boolean('FOC', 'Wave to ocean energy flux [FOX]', default=False)
    wave_ocean.add_boolean('TUS', 'Stokes transport [TUS]', default=False)
    wave_ocean.add_boolean('USS', 'Surface Stokes drift [USS]', default=False)
    wave_ocean.add_boolean('P2S', 'Second-order sum pressure [P2S]', default=False)
    wave_ocean.add_boolean('USF', 'Spectrum of surface sStokes drift [USF]', default=False)
    wave_ocean.add_boolean('P2L', 'Micro seism source term [P2L]', default=False)
    wave_ocean.add_boolean('TWI', 'Wave to sea ice stress [TWI]', default=False)
    wave_ocean.add_boolean('FIC', 'Wave to sea ice energy flux [FIC]', default=False)
    wave_ocean.add_boolean('USP', 'Pertitioned surface Stokes drift [USP]', default=False)

    # Wave-bottom layer (advanced)
    wave_bottom = gm.model_parameters.add_group('wave_bottom', 'Wave-bottom layer (advanced)')

    wave_bottom.add_boolean('ABR', 'Near bottom rms aplitides [ABR]', default=False)
    wave_bottom.add_boolean('UBR', 'Near bottom rms velocities [UBR]', default=False)
    wave_bottom.add_boolean('BED', 'Bedforms [BED]', default=False)
    wave_bottom.add_boolean('FBB', 'Energy flux due to bottom friction [FBB]', default=False)
    wave_bottom.add_boolean('TBB', 'Momentum flux due to bottom friction [TBB]', default=False)

    # Spectrum parameters (advanced)
    spectrum_parameters = gm.model_parameters.add_group('spectrum_parameters', 'Spectrum parameters (advanced)')

    spectrum_parameters.add_boolean('MSS', ' Mean square slopes [MSS]', default=False)
    spectrum_parameters.add_boolean('MSC', ' Spectral tail level [MSC]', default=False)
    spectrum_parameters.add_boolean('WL02', 'East/X North/Y mean wavelength comp [WL02]', default=False)
    spectrum_parameters.add_boolean('AXT', 'Correl sea surface gradients (x,t) [AXT]', default=False)
    spectrum_parameters.add_boolean('AYT', 'Correl sea surface gradients (y,t) [AYT]', default=False)
    spectrum_parameters.add_boolean('AXY', 'Correl sea surface gradients (x,y) [AXY]', default=False)

    # Numerical diagnostics (advanced)
    numerical_diagnostics = gm.model_parameters.add_group('numerical_diagnostics', 'Numerical diagnostics (advanced)')

    numerical_diagnostics.add_boolean('DTD', 'Average time step in integration [DTD]', default=False)
    numerical_diagnostics.add_boolean('FC', 'Cut-off frequency [FC]', default=False)
    numerical_diagnostics.add_boolean('CFX', 'Max. CFL number for spatial advection [CFX]', default=False)
    numerical_diagnostics.add_boolean('CFD', 'Max. CFL number for theta-advection [CFD]', default=False)
    numerical_diagnostics.add_boolean('CFK', 'Max CFL number for k-advection [CFK]', default=False)

    # User defined (advanced)
    user_defined = gm.model_parameters.add_group('user_defined', 'User defined (advanced)')

    u1 = user_defined.add_boolean('U1', 'User defined #1 (requires coding) [U1]', default=False)
    user_defined.add_boolean('U2', 'User defined #2 (requires coding) [U2]', default=False)

    # OUNF post-processing - these are not accessible through the model control dialog
    hidden = {True: False, False: False}
    user_defined.add_text('field_timestart', '').add_dependency(u1, hidden)
    user_defined.add_integer('field_timestride', '', 0).add_dependency(u1, hidden)
    user_defined.add_integer('field_timecount', '', 1_000_000_000).add_dependency(u1, hidden)
    user_defined.add_integer('field_timesplit', '', 2).add_dependency(u1, hidden)
    user_defined.add_text('field_list', '').add_dependency(u1, hidden)
    user_defined.add_integer('field_samefile', '', 1).add_dependency(u1, hidden)
    user_defined.add_integer('field_vector', '', 1).add_dependency(u1, hidden)
    user_defined.add_integer('field_type', '', 1).add_dependency(u1, hidden)
    user_defined.add_text('file_prefix', '', 'ww3.').add_dependency(u1, hidden)
    user_defined.add_integer('file_netcdf', '', 4).add_dependency(u1, hidden)
    user_defined.add_integer('file_ix0', '', 1).add_dependency(u1, hidden)
    user_defined.add_integer('file_ixn', '', 1_000_000_000).add_dependency(u1, hidden)
    user_defined.add_integer('file_iy0', '', 1).add_dependency(u1, hidden)
    user_defined.add_integer('file_iyn', '', 1_000_000_000).add_dependency(u1, hidden)

    return gm
